#!perl
#**************************************************************************
#
# ACTINIC.pm	- module for common functions among the Actinic scripts
#
# Written by George Menyhert
#
# Copyright (c) Actinic Software Ltd 1998
#
#**************************************************************************

package ACTINIC;
require 5.002;

push (@INC, "cgi-bin");
NETQUOTEVAR:INCLUDEPATHADJUSTMENT

require NETQUOTEVAR:ACTINICSAFER;
require NETQUOTEVAR:ACTINICDIFFIE;
require NETQUOTEVAR:ACTINICENCRYPT;

use Socket;
use strict;
 
#
# define some global constants
#
$::FALSE 	= 0;											# return codes
$::TRUE	 	= 1;

$::FAILURE 	= 0;
$::SUCCESS 	= 1;
$::NOTFOUND = 2;
$::FAILEDSEARCH = $::NOTFOUND;						# synonyms
$::EOF		= 3;
$::EOB     	= 4;
$::BADDATA	= 5;
$::WARNING	= 6;
$::ACCEPTED	= 7;
$::REJECTED	= 8;
$::PENDING	= 9;

$::VARPREFIX 	= 'NETQUOTEVAR:';						# template variables
$::DELPREFIX 	= 'NETQUOTEDEL:';						# template delimiters

$::RBBYTE     = 0;										# enumeration of the field types
$::RBWORD     = 1;
$::RBDWORD    = 2;
$::RBQWORD    = 3;
$::RBSTRING   = 4;
$::RBKEY      = 5;

$::HIDDEN	= 0;											# the prompt status
$::OPTIONAL	= 1;
$::REQUIRED = 2;

$::PAYMENT_CREDIT_CARD				= 0;				# the various payment methods
$::PAYMENT_CASH_ON_DELIVERY		= 1;
$::PAYMENT_CHECK_ON_DELIVERY		= 2;
$::PAYMENT_INVOICE					= 3;
$::PAYMENT_INVOICE_PRE_PAY			= 4;
$::PAYMENT_CREDIT_CARD_SEPARATE	= 5;

$::ORDER_AID_NONE			= 0;							# the order aid methods
$::ORDER_AID_COMPLETE	= 1;
$::ORDER_AID_RESPOND		= 2;
$::ORDER_AID_CONTINUE	= 3;

$::LOCK_SH = 1;											# flock - share permissions
$::LOCK_EX = 2;											# flock - exclusive lock
$::LOCK_NB = 4;											# flock - non-blocking (can be or'ed with others)
$::LOCK_UN = 8;											# flock - unlock

$::g_sRequiredColor	= '#aa3333';					# the "required" field color

$::g_sCancelButtonLabel = '';							# the global button labels
$::g_sConfirmButtonLabel = '';
$::g_sAddToButtonLabel = '';
$::g_sEditButtonLabel = '';
$::g_sRemoveButtonLabel = '';
$::g_sSearchButtonLabel = '';

$::s_nErrorRecursionCounter = 0;

umask (0177);												# update the process umask
	
#
# define some ACTINIC package constants
#

$ACTINIC::prog_name = 'ACTINIC.pm';					# Program Name 
$ACTINIC::prog_name = $ACTINIC::prog_name;		# remove compiler warning
$ACTINIC::prog_ver = '$Revision: 255 $ ';				# program version
$ACTINIC::prog_ver = substr($ACTINIC::prog_ver, 11); # strip the revision information
$ACTINIC::prog_ver =~ s/ \$//;						# and the trailers

$ACTINIC::BILLCONTACT 	= "INVOICE";
$ACTINIC::SHIPCONTACT 	= "DELIVERY";
$ACTINIC::SHIPINFO 		= "SHIPPING";
$ACTINIC::TAXINFO 		= "TAX";
$ACTINIC::GENERALINFO 	= "GENERAL";
$ACTINIC::PAYMENTINFO 	= "PAYMENT";
$ACTINIC::LOCATIONINFO 	= "LOCATION";

$ACTINIC::FILE				= 0;
$ACTINIC::SDTOUT			= 1;
$ACTINIC::MEMORY			= 2;

$ACTINIC::s_bTraceSocket = $::FALSE;
$ACTINIC::s_bTraceSockFirstPass = $::TRUE;
$ACTINIC::s_bTraceFileFirstPass = $::TRUE;

$ACTINIC::ORDER_BLOB_MAGIC = hex('10');
$ACTINIC::ORDER_DETAIL_BLOB_MAGIC = hex("11");

$ACTINIC::FORM_URL_ENCODED 			= 0;			# standard application/x-www-form-urlencoded (%xx) encoding
$ACTINIC::MODIFIED_FORM_URL_ENCODED	= 1;			# Actinic format - identical to eParameter except an
																# underscore is used instead of a percent sign and the string is
																# prepended with an "a"

$ACTINIC::B2B = new ACTINIC_B2B();					# Create B2B object to keep B2B parameters
$ACTINIC::USESAFE = $::TRUE;							# If true we attempt to use Safe.pm
$ACTINIC::USESAFEONLY = $::FALSE;					# If true, eval is only allowed in Safe.pm

$ACTINIC::MAX_RETRY_COUNT      = 10;
$ACTINIC::RETRY_SLEEP_DURATION = 1;
$ACTINIC::DOS_SLEEP_DURATION = 2;

$ACTINIC::AssertIsActive = $::FALSE;				# true if an assert is being reported
$ACTINIC::AssertIsLooping = $::FALSE;				# true if the assert function appears to be stuck in a loop

#######################################################
#
# GetActinicDate - Get the current date in Actinic
#	format (GMT server time)
#
# Returns: 	the date in YYYY/MM/DD HH:MM format
#
#######################################################

sub GetActinicDate
	{
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	#
	# Get the current date/time on the server
	#
	my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst, $sDate);
	($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = gmtime(time);	# platform independent time
	$mon++;													# make month 1 based
	$year += 1900;											# make year AD based
	$sDate = sprintf("%4.4d/%2.2d/%2.2d %2.2d:%2.2d", $year, $mon, $mday, $hour, $min);
	#
	# Misc info
	#
	return($sDate);										# the date
	}

#######################################################
#
# InitMonthMap - initialize the month maps.  This
#	subroutine must be called after ReadPromptFile.
#
# Affects: 	%::g_MonthMap (hash table mapping month names
#					to their numbers
#				%::g_InverseMonthMap - hash table inversion
#					of %::g_MonthMap
#
#######################################################

sub InitMonthMap
	{
	%::g_MonthMap = (GetPhrase(-1, 0), 1,			# hash to convert month to digit
						GetPhrase(-1, 1), 2,
						GetPhrase(-1, 2), 3,
						GetPhrase(-1, 3), 4,
						GetPhrase(-1, 4), 5,
						GetPhrase(-1, 5), 6,
						GetPhrase(-1, 6), 7,
						GetPhrase(-1, 7), 8,
						GetPhrase(-1, 8), 9,
						GetPhrase(-1, 9), 10,
						GetPhrase(-1, 10), 11,
						GetPhrase(-1, 11), 12);
	my ($key, $value);
	while ( ($key, $value) = each %::g_MonthMap)	# build a revers map
		{
		$::g_InverseMonthMap{$value} = $key;
		}
	}

#######################################################
#
# GetCountryName - map the country code to country name
#
# Params:	0 - country code
#
# Returns:	0 - country name or undef on error
#
#######################################################

sub GetCountryName
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in CountryName ($#_)", __LINE__, __FILE__);
#? ACTINIC::ASSERT(defined $::g_pLocationList, "Location list undefined", __LINE__, __FILE__);
	my $sCode = $_[0];
	return ($$::g_pLocationList{$sCode});
	}

#######################################################
#
# GetHostname - attempt to retrieve the hostname
#
#	Returns:	0 - hostname or IP address or ''
#
#######################################################

sub GetHostname
	{
	my $sLocalhost = $ENV{SERVER_NAME};				# try the environment
	$sLocalhost =~ s/[^-a-zA-Z0-9.]//g;				# strip any bad characters

	if (!$sLocalhost)										# if still no hostname is found
		{
		$sLocalhost = $ENV{HOST};						# try a different environment variable
		$sLocalhost =~ s/[^-a-zA-Z0-9.]//g;			# strip any bad characters
		}
	if (!$sLocalhost)										# if still no hostname is found
		{
		$sLocalhost = $ENV{HTTP_HOST};				# try a different environment variable
		$sLocalhost =~ s/[^-a-zA-Z0-9.]//g;			# strip any bad characters
		}
	if (!$sLocalhost)										# if still no hostname is found
		{
		$sLocalhost = $ENV{LOCALDOMAIN};				# try a different environment variable
		$sLocalhost =~ s/[^-a-zA-Z0-9.]//g;			# strip any bad characters
		}
	if (!$sLocalhost)										# if still no hostname is found
		{
		$sLocalhost = `hostname`;						# try the command line
		$sLocalhost =~ s/[^-a-zA-Z0-9.]//g;			# strip any bad characters
		}
	if (!$sLocalhost &&									# if still no hostname and 
		 $^O eq 'MSWin32')								# NT
		{
		my $sHost = `ipconfig`;							# run ipconfig and gather the collection of addresses
		$sHost =~ /IP Address\D*([0-9.]*)/;			# get the first address in the list
		$sLocalhost = $1;
		$sLocalhost =~ s/[^-a-zA-Z0-9.]//g;			# strip any bad characters
		}

	return ($sLocalhost);
	}
	
#######################################################
#
# SendMail - Send an email to the specified email
#	address if this service has been requested.
#
#	Params:	0 - the smtp server ip address
#				1 - the destination email address
#				2 - the subject
#				3 - the message
#           4 - optional return address
#
#	Returns:	0 - status
#				1 - message
#
#######################################################

sub SendMail
	{
#? ACTINIC::ASSERT($#_ >= 3, "Invalid argument count in SendMail ($#_)", __LINE__, __FILE__);

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	if ($#_ < 3)
		{
		return($::FAILURE, GetPhrase(-1, 12, 'Actinic::SendMail'), 0, 0);
		}
	
	my ($sSmtpServer, $sEmailAddress, $sSubjectText, $sMessageText, $sReturnAddress) = @_;
	#
	# pass it on to the rich mail function
	#
	return(SendRichMail($sSmtpServer, $sEmailAddress, $sSubjectText, $sMessageText, "", $sReturnAddress));
	}

#######################################################
#
# SendRichMail - Send an email to the specified email
#	address if this service has been requested.
#
#	Params:	0 - the smtp server ip address
#				1 - the destination email address
#				2 - the subject
#				3 - the message as text
#				4 - the message as HTML
#           5 - optional return address
#
#	Returns:	0 - status
#				1 - message
#
#######################################################

sub SendRichMail
	{
#? ACTINIC::ASSERT($#_ >= 4, "Invalid argument count in SendRichMail ($#_)", __LINE__, __FILE__);

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	if ($#_ < 4)
		{
		return($::FAILURE, GetPhrase(-1, 12, 'Actinic::SendRichMail'), 0, 0);
		}
	
	my ($sSmtpServer, $sEmailAddress, $sLocalError, $sSubjectText, $sMessageText, $sMessageHTML, $sBoundary, $sReturnAddress);
	($sSmtpServer, $sEmailAddress, $sSubjectText, $sMessageText, $sMessageHTML, $sReturnAddress) = @_;
	if (!$sReturnAddress)								# if no return address defined
		{
		$sReturnAddress = $sEmailAddress;			# use the destination email address
		}
	#
	# Gather the SMTP host, server, and socket information
	#
	my ($nProto, $them, $nSmtpPort, $sLocalHost, $sMessage, $serverIP);
	
	my $sLocalhost = GetHostname();						# get the local machine ip address
	if ($sLocalhost eq '')
		{
		$sLocalhost = 'www.actinic.com';
		}
	
	$nProto = getprotobyname('tcp');
	$nSmtpPort = 25;										# Use default port
	
	$serverIP = inet_aton($sSmtpServer);			# due the dns lookup and get the ip address	
	if (!defined $serverIP)
		{
		return($::FAILURE, GetPhrase(-1, 13, $!), 0, 0); # Record internal error 
		}
	
	$them = sockaddr_in($nSmtpPort, $serverIP);	# create the sockaddr
	if (!defined $them)
		{
		return($::FAILURE, GetPhrase(-1, 14, $!), 0, 0); # Record internal error 
		}
	
	unless (socket(MYSOCKET, PF_INET, SOCK_STREAM, $nProto))	# create the socked
		{
		return($::FAILURE, GetPhrase(-1, 15, $!), 0, 0); # Record internal error 
		}
	
	unless (connect(MYSOCKET, $them))				# connect to the remote host
		{
		$sLocalError = GetPhrase(-1, 16, $!);		# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	binmode MYSOCKET;										# just incase
	
	my($oldfh) = select(MYSOCKET);					# make MYSOCKET the current file handle
	$| = 1;													# make each command send a flush
	select($oldfh);										# return to the default file handle
	
	$sMessage = <MYSOCKET>;								# see what the SMTP server has to say
	if ($sMessage =~ /^[45]/)							# check for failures from the SMTP server
		{
		$sLocalError = GetPhrase(-1, 17, 1, $sMessage);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	unless (print MYSOCKET "HELO $sLocalhost\r\n")	# start the conversation with the SMTP server
		{
		$sLocalError = GetPhrase(-1, 18, 1, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	$sMessage = <MYSOCKET>;								# see what the SMTP server has to say
	if ($sMessage =~ /^[45]/)							# check for failures
		{
		$sLocalError = GetPhrase(-1, 17, 2, $sMessage);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	unless (print MYSOCKET "MAIL FROM:<" . $sReturnAddress . ">\r\n") # specify the origin
		{
		$sLocalError = GetPhrase(-1, 18, 2, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	$sMessage = <MYSOCKET>;								# see what the SMTP server has to say
	if ($sMessage =~ /^[45]/)							# check for failures
		{
		$sLocalError = GetPhrase(-1, 17, 3, $sMessage);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	unless (print MYSOCKET "RCPT TO:<",$sEmailAddress,">\r\n") # reciepient is always the supplier
		{
		$sLocalError = GetPhrase(-1, 18, 3, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	$sMessage = <MYSOCKET>;								# see what the SMTP server has to say
	if ($sMessage =~ /^[45]/)							# check for failures
		{
		$sLocalError = GetPhrase(-1, 17, 4, $sMessage);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	unless (print MYSOCKET "DATA\r\n")				# the rest of the is the message body until the <CRLF>.<CRLF>
		{
		$sLocalError = GetPhrase(-1, 18, 4, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	$sMessage = <MYSOCKET>;								# see what the SMTP server has to say
	if ($sMessage =~ /^[45]/)							# check for failure
		{
		$sLocalError = GetPhrase(-1, 17, 5, $sMessage);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}

	if ($sMessageText ne '' && $sMessageHTML ne '')# if both messages are specified
		{
		#
		# make up our multi-part boundary from the order number
		#
		$sBoundary = "------------" . $::g_InputHash{ORDERNUMBER};
		#
		# let server know we are sending MIME
		#
		unless (print MYSOCKET "MIME-Version: 1.0\r\n") # MIME version
			{
			$sLocalError = GetPhrase(-1, 18, 11, $!);	# Record internal error 
			close MYSOCKET;
			return($::FAILURE, $sLocalError, 0, 0);
			}
		}
	else														# this isn't a multi-part message
		{
		$sBoundary = "";									# clear the boundary
		}

	unless (print MYSOCKET "From: $sReturnAddress\r\n") # subject
		{
		$sLocalError = GetPhrase(-1, 18, 5, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	unless (print MYSOCKET "Subject: $sSubjectText\r\n") # subject
		{
		$sLocalError = GetPhrase(-1, 18, 6, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	unless (print MYSOCKET "To: $sEmailAddress\r\n") # subject
		{
		$sLocalError = GetPhrase(-1, 18, 7, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	unless (print MYSOCKET "Reply-To: $sReturnAddress\r\n") # subject
		{
		$sLocalError = GetPhrase(-1, 18, 8, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	if ($sBoundary ne '')								# if both message types are specified
		{
		my $sContentMultipart = "Content-Type: multipart/alternative; ";
		$sContentMultipart .= "boundary=\"" . $sBoundary . "\"\r\n\r\n";

		unless (print MYSOCKET $sContentMultipart) # content-type
			{
			$sLocalError = GetPhrase(-1, 18, 12, $!);	# Record internal error 
			close MYSOCKET;
			return($::FAILURE, $sLocalError, 0, 0);
			}
		}

	unless (print MYSOCKET "\r\n")					# blank line
		{
		$sLocalError = GetPhrase(-1, 18, 8, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	if ($sBoundary ne '')								# if both message types are specified
		{
		#
		# send the text multipart 
		#
		my $sTextMultipart = "--" . $sBoundary . "\r\n";
		$sTextMultipart .= "Content-Type: text/plain; charset=us-ascii\r\n";
		$sTextMultipart .= "Content-Transfer-Encoding: 7bit\r\n\r\n" . $sMessageText . "\r\n\r\n";

		unless (print MYSOCKET $sTextMultipart)	# text content
			{
			$sLocalError = GetPhrase(-1, 18, 13, $!);	# Record internal error 
			close MYSOCKET;
			return($::FAILURE, $sLocalError, 0, 0);
			}

		#
		# send the HTML multipart 
		#
		my $sHTMLMultipart = "--" . $sBoundary . "\r\n";
		$sHTMLMultipart .= "Content-Type: text/html; charset=us-ascii\r\n";
		$sHTMLMultipart .= "Content-Transfer-Encoding: 7bit\r\n\r\n" . $sMessageHTML . "\r\n\r\n";

		unless (print MYSOCKET $sHTMLMultipart)	# HTML content
			{
			$sLocalError = GetPhrase(-1, 18, 14, $!);	# Record internal error 
			close MYSOCKET;
			return($::FAILURE, $sLocalError, 0, 0);
			}

		#
		# send the final boundary 
		#
		my $sEndMultipart = "--" . $sBoundary . "--\r\n";
		unless (print MYSOCKET $sEndMultipart)		# multipart terminator
			{
			$sLocalError = GetPhrase(-1, 18, 15, $!);	# Record internal error 
			close MYSOCKET;
			return($::FAILURE, $sLocalError, 0, 0);
			}
		}
	else
		{
		unless (print MYSOCKET "$sMessageText\r\n")	# just spacing
			{
			$sLocalError = GetPhrase(-1, 17, 6, $sMessage); # Record internal error 
			close MYSOCKET;
			return($::FAILURE, $sLocalError, 0, 0);
			}
		}
	unless (print MYSOCKET "\r\n.\r\n")				# finish the message
		{
		$sLocalError = GetPhrase(-1, 18, 9, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	$sMessage = <MYSOCKET>;								# see what the SMTP server has to say
	if ($sMessage =~ /^[45]/)							# check for failures
		{
		$sLocalError = GetPhrase(-1, 17, 7, $sMessage);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	unless (print MYSOCKET "QUIT\r\n")					# end the conversation
		{
		$sLocalError = GetPhrase(-1, 18, 10, $!);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	$sMessage = <MYSOCKET>;								# see what the SMTP server has to say
	if ($sMessage =~ /^[45]/)							# check for failures
		{
		$sLocalError = GetPhrase(-1, 17, 8, $sMessage);	# Record internal error 
		close MYSOCKET;
		return($::FAILURE, $sLocalError, 0, 0);
		}
	
	shutdown MYSOCKET, 1;								# shutdown sends
	close MYSOCKET;										# done
	
	return($::SUCCESS, '', 0, 0);
	}

#######################################################
#																		
# GetCookie - retrieve the actinic cookie
#
# Returns:	0 - cookie (undef if undefined)
#
#######################################################

sub GetCookie
	{
	my ($sCartID, $sContactDetails) = GetCookies();
	return ($sCartID);
	}
	
#######################################################
#																		
# GetCookies - retrieve the actinic cookies
#
# Returns:	0 - cart ID (undef if undefined)
#				1 - checkout details (undef if undefined)
#
#######################################################

sub GetCookies
	{
	my ($sCookie, $sCookies);
	$sCookies = $::ENV{'HTTP_COOKIE'};				# try to retrieve the cookie
	my (@CookieList) = split(/;/, $sCookies);		# separate the various cookie variables in the list
	my ($sLabel);
	my $bFound = $::FALSE;								# true when one of the cookies has been found
	my ($sCartID, $sContactDetails);
	my ($sDigest,$sBaseFile) = ACTINIC::CaccGetCookies();
	my $bIgnoreContact = $sDigest || $::g_InputHash{HASH};
	foreach $sCookie (@CookieList)
		{
		$sCookie =~ s/^\s*//;							# strip leading white space
		if ($sCookie =~ /^ACTINIC_CART/)				# found the cart ID
			{
			($sLabel, $sCartID) = split (/=/, $sCookie);	# retrieve the value
			#
			# Make the cart ID secure by locking out any shell type characters
			#
			$sCartID =~ /([a-zA-Z0-9]+)/;				# cart ID's are just characters
			$sCartID = $1;

			if ($bIgnoreContact or $bFound)			# if the other cookie has already been found
				{
				last;											# exit the loop
				}
			else												# this is the first of the two cookies to be found
				{
				$bFound = $::TRUE;						# note that we found it
				}
			}
		elsif (!$bIgnoreContact and $sCookie =~ /^ACTINIC_CONTACT/)		# found the contact details
			{
			($sLabel, $sContactDetails) = split (/=/, $sCookie);	# retrieve the value
			#
			# strip any trailing or leading quotes and spaces
			#
			$sContactDetails =~ s/^\s*"?//;        # " # here for emacs formatting
			$sContactDetails =~ s/"?\s*$//;        # " # here for emacs formatting

			if ($bFound)									# if the other cookie has already been found
				{
				last;											# exit the loop
				}
			else												# this is the first of the two cookies to be found
				{
				$bFound = $::TRUE;						# note that we found it
				}
			}
		}
	return ($sCartID, $sContactDetails);
	}
	
#######################################################
#																		
# GetReferrer - retrieve the referrer URL
#
# Returns:	0 - referring URL
#
#######################################################

sub GetReferrer
	{
# This is not strictly necessary and causes problems with autotrial and host #? ACTINIC::ASSERT(defined %::g_InputHash, "g_InputHash is undefined in GetReferrer", __LINE__, __FILE__);
	my ($sURL);
	$sURL = $::ENV{"HTTP_REFERER"};					# try to retrieve the cookie
	if (defined %::g_InputHash &&
		 defined $::g_InputHash{ACTINIC_REFERRER})
		{
		$sURL = $::g_InputHash{ACTINIC_REFERRER};
		if( $sURL !~ /\/$/ )								# Make sure that if URL is a directory it ends with '/'
			{
			my @fields = split('/',$sURL);			# Split it into fields
			my $fnam = pop @fields;						# Get last field
			if( $fnam !~ /\./ )							# Check if it looks like a file name
				{
				$sURL .= '/';								# Looks like a directory without trailing '/', add it
				}
			}
		}
	#
	# If we still don't have referrer then use the hard coded base URL
	#
	if ($sURL eq "/" || $sURL eq "")
		{
		#
		# Ensure that the setup blob is already loaded
		#
		if (!defined $::g_pSetupBlob ||
		 	 scalar(keys(%$::g_pSetupBlob)) == 0)
			{
			my $sPath = ACTINIC::GetPath();			# get the path to the web site
			ACTINIC::ReadSetupFile($sPath);			# read the setup
			}
		#
		# Use the hard coded URL
		#
		$sURL = $$::g_pSetupBlob{CATALOG_URL};
		}	
	return ($sURL);
	}

#######################################################
#																		
# TrimHashEntries - trim leading and trailing white
#	space from every value in the hash table
#
# Params:	0 - in/out - pointer to the hash
#
#######################################################

sub TrimHashEntries
	{
#? ACTINIC::ASSERT(0 == $#_, "Invalid parameter count in TrimHashEntries, $#_", __LINE__, __FILE__);
	my $pHash = $_[0];
	#
	# process each entry in the hash
	#
	my ($key, $value);
	while ( ($key, $value) = each %$pHash)
		{
		$$pHash{$key} =~ s/^\s*(.*?)\s*$/$1/gs;
		}
	}

#######################################################
#
# UUEncode - Returns a base 64 encoded string
#
# Params:	[0] - $sInputString
#
# Returns:	($sOutput) - Encoded string
#
#######################################################

sub UUEncode
	{
	my ($sInput) = @_;
	my $sOutput = "";
	my ($i, $cByte, $nByteNo, $nLeftOver);
	my @arrInput = unpack("C*", $sInput);
	use integer;
	#
	# Lookup table for output characters
	#
	my $sLookup = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

	$nByteNo = 0;
	foreach $cByte (@arrInput)
		{
		if($nByteNo == 0)
			{
			#
			# The first input byte just takes the top 6 bits
			# and then passes on the lower 2 bits to be the
			# top 2 bits of the next byte
			#
			$sOutput .= substr($sLookup, ($cByte >> 2) & 63, 1);
			$nLeftOver = ($cByte << 4) & 48;			# Leave 2 bits
			$nByteNo++;
			}
		elsif($nByteNo == 1)
			{
			#
			# The second input byte takes the top 4 bits
			# and makes them the lower 4 bits, combined with
			# the two bits from the first byte
			#
			$sOutput .= substr($sLookup, $nLeftOver | (($cByte >> 4) & 15), 1);
			$nLeftOver = ($cByte << 2) & 60;			# Keep lower 4 bits (shifted up)
			$nByteNo++;
			}
		elsif($nByteNo == 2)
			{
			#
			# Finally we take the left over 4 bits as the
			# upper 4 bits, together with 2 bits from
			# the top of the new byte. The last 6
			# bits can be used as-is
			#
			$sOutput .= substr($sLookup, $nLeftOver | (($cByte >> 6) & 3), 1);
			$sOutput .= substr($sLookup, $cByte & 63, 1);		# Take last 6 bits
			$nByteNo = 0;
			}
		}
	if($nByteNo == 1)
		{
		$sOutput .= substr($sLookup, $nLeftOver, 1);
		$sOutput .= '==';
		}
	elsif($nByteNo == 2)
		{
		$sOutput .= substr($sLookup, $nLeftOver, 1);
		$sOutput .= '=';
		}
	return($sOutput);
	}

##################################################################################
#																											#
# HTML manipulation functions - begin															#
#																											#
##################################################################################

#######################################################
#																		
# ProcessEscapableText - encode the text from the
#	specified string leaving escaped regions raw.
#
# Params:	0 - the string to convert
#
# Returns:	0 - status
#				1 - modified string or error message (if any)
#				2 - 0
#				3 - 0
#
#######################################################

sub ProcessEscapableText
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ProcessEscapableText ($#_)", __LINE__, __FILE__);

	my ($sString) = @_;
	#
	# first see if there is any escaped text
	#
	my (@Response);
	if ($sString !~ /!!</)								# no escaped text
		{
		return (EncodeText($sString));				# encode it
		}
	#
	# pick apart the string
	#
	my (@PartsList) = ($sString =~  m/((.*?)!!<(.*?)>!!)*/g);
	my ($sEndPart) = ($sString =~ m/>!!(.*?)$/g); # get the closing encode text
	#
	# Now @PartsList contains a series of segments of the following pattern:
	#
	#		element		description
	#			0			the entire segment - throw out
	#			1			text to encode
	#			2			raw HTML
	#
	my ($sPart, $sNewString, $nCount, $nElement);
	$nCount = 0;
	foreach $sPart (@PartsList)
		{
		$nElement = Modulus($nCount, 3);			# calculate the element number
		
		if ($nElement == 0)								# the entire segment
			{
			# no-op - throw out
			}
		elsif ($nElement == 1)							# text to be encoded
			{
			@Response = EncodeText($sPart);			# encode it
			if ($Response[0] != $::SUCCESS)
				{
				return (@Response);
				}
			$sNewString .= $Response[1];
			}
		elsif ($nElement == 2)							# raw HTML
			{
			$sNewString .= $sPart;
			}
		$nCount++;
		}
	#
	# the end part needs to be encoded and included
	#
	@Response = EncodeText($sEndPart);				# encode it
	if ($Response[0] != $::SUCCESS)
		{
		return (@Response);
		}
	$sNewString .= $Response[1];						# and include it
	
	return ($::SUCCESS, $sNewString, 0, 0);
	}
	
#######################################################
#																		
# EncodeText2 - convert then non-alphanumeric characters in
#	the supplied string to &#xx; where xx is the
#	equivalent decimal code for the character.  This is
#	needed for the HTML printout
#
# Params:	0 - the string to convert
#				1 - (optional) if TRUE, do HTML encoding (&#d;)
#					if FALSE, do CGI encodeing (%x).  Default - TRUE
#				2 - (optional) if TRUE make spaces &nbsp;,
#					default - FALSE.  Only makes sense in
#					the context of 1 = TRUE
#
# Returns:	0 - modified string
#
#######################################################

sub EncodeText2
	{
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my @Response = EncodeText(@_);
#? ACTINIC::ASSERT($Response[0] == $::SUCCESS, "It looks like EncodeText can return an error.", __LINE__, __FILE__);
	return ($Response[1]);
	}
	
#######################################################
#																		
# EncodeText - convert then non-alphanumeric characters in
#	the supplied string to &#xx; where xx is the
#	equivalent decimal code for the character.  This is
#	needed for the HTML printout
#
# Params:	0 - the string to convert
#				1 - (optional) if TRUE, do HTML encoding (&#d;)
#					if FALSE, do CGI encodeing (%x).  Default - TRUE
#				2 - (optional) if TRUE make spaces &nbsp;,
#					default - FALSE.  Only makes sense in
#					the context of 1 = TRUE
#
# Returns:	0 - status
#				1 - modified string or error message (if any)
#				2 - 0
#				3 - 0
#
#######################################################

sub EncodeText
	{
#? ACTINIC::ASSERT($#_ >= 0, "Invalid argument count in EncodeText ($#_)", __LINE__, __FILE__);
	
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sString, $bHtmlEncoding, $bNBSP) = @_;
	if (!defined $bHtmlEncoding)						# default encoding is HTML
		{
		$bHtmlEncoding = $::TRUE;
		}
	if (!defined $bNBSP)									# default NBSP is FALSE
		{
		$bNBSP = $::FALSE;
		}
	#
	# Do the substitution.
	#
	if ($bHtmlEncoding)									# HTML encoding
		{
		$sString =~ s/(\W)/sprintf('&#%d;', ord($1))/eg;	# regular space substitution
		}
	else														# CGI encoding
		{
		$sString =~ s/(\W)/sprintf('%%%2.2x', ord($1))/eg;	# regular space substitution
		}
	
	if ($bNBSP)												# if we want non-breaking spaces
		{
		$sString =~ s/&#32;/&nbsp;/g;					# replace the normal spaces with the non-breaking versions
		}														# NOTE: this does nothing if ! $bHtmlEncoding
		
	return ($::SUCCESS, $sString, 0, 0);
	}

#######################################################
#																		
# DecodeText - this function is similar
#	to EncodeText with two exceptions: 1) it deals with
#	characters stored as %xx and 2) it works in reverse
#	restoring the character for the % value
#
# Params:	0 - the string to convert
#				1 - decode method flag $ACTINIC::FORM_URL_ENCODED or $ACTINIC::MODIFIED_FORM_URL_ENCODED
#					$ACTINIC::FORM_URL_ENCODED = decode using application/x-www-form-urlencoded (%xx)
#					$ACTINIC::MODIFIED_FORM_URL_ENCODED = Actinic format - identical to $::FORM_URL_ENCODED except an
#						underscore is used instead of a percent sign and the string is
#						prepended with an "a".  This encoding is used to map arbitrary
#						strings into HTML "ID and NAME" data types.
#						NAME tokens must begin with a letter ([A-Za-z]) and may be
#						followed by any number of letters, digits ([0-9]), hyphens ("-"),
#						underscores ("_"), colons (":"), and periods (".")
#
# Returns:	($sString) - the converted string
#
#######################################################

sub DecodeText
	{
#? ACTINIC::ASSERT($#_ == 1, "Invalid argument count in DecodeText ($#_)", __LINE__, __FILE__);

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sString, $eEncoding) = @_;
	
	if ($eEncoding == $ACTINIC::MODIFIED_FORM_URL_ENCODED)
		{
		$sString =~ s/^a//;								# string the leading a
		$sString =~ s/_([A-Fa-f0-9]{2})/pack('c',hex($1))/ge;	# Convert _XX from hex numbers to character equivalent
		}
	elsif ($eEncoding == $ACTINIC::FORM_URL_ENCODED)
		{
		$sString =~ s/\+/ /g;							# replace + signs with the spaces they represent		
		$sString =~ s/%([A-Fa-f0-9]{2})/pack('c',hex($1))/ge;	# Convert %XX from hex numbers to character equivalent
		}
	else
		{
#? ACTINIC::ASSERT($::FALSE, 'Invalid encodgin argument to DecodeText' . " ($eEncoding)", __LINE__, __FILE__);
		}
	
	return ($sString);
	}

#######################################################
#																		
# TemplateFile - replace the vars in the template file
#	with the values stored in the variable table
#
# Params:	0 - template filename
#				1 - a reference to the variable table
#
# Returns:  0 - $::SUCCESS or $::FAILURE on error
#				1 - error message
#				2 - modified HTML
#				3 - 0
#
#######################################################

sub TemplateFile
	{
#? ACTINIC::ASSERT($#_ >= 0, "Invalid argument count in TemplateFile ($#_)", __LINE__, __FILE__);
	
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sFilename, $pVariableTable);
	($sFilename, $pVariableTable) = @_;
	
	unless (open (TFFILE, "<$sFilename"))
		{
		return($::FAILURE, GetPhrase(-1, 21, $sFilename, $!), '', 0);
		}
		
	my ($sOutput);
	{
	local $/;
	$sOutput = <TFFILE>;								# read the entire file
	}
	close (TFFILE);
	
	return (TemplateString($sOutput, $pVariableTable));
	}

#######################################################
#																		
# TemplateString - replace the vars in the template
#	string with their values
#
# Params:	0 - template string
#				1 - a reference to the variable table
#
# Returns:  0 - $::SUCCESS or $::FAILURE on error
#				1 - error message
#				2 - modified HTML
#				3 - 0
#
#######################################################

sub TemplateString
	{
#? ACTINIC::ASSERT($#_ == 1, "Invalid argument count in TemplateString ($#_)", __LINE__, __FILE__);
	
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sString, $pVariableTable);
	($sString, $pVariableTable) = @_;
	
	if(defined $$pVariableTable{'NETQUOTEVAR:ADVANCEDTAXHTML'})
		{
		my %hashEmpty = {};
		my @Response = TemplateFile(GetPath()."advancedtax.html", \%hashEmpty);
		if ($Response[0] != $::SUCCESS)
			{
			return (@Response);
			}
		$sString =~ s/(NETQUOTEDEL:TAXPHASE)(.*?)NETQUOTEVAR:TAXPROMPT.*?NETQUOTEDEL:TAXPHASE/$1$Response[2]$1/isg;				# replace the variable with its value
		delete $$pVariableTable{'NETQUOTEVAR:ADVANCEDTAXHTML'};
		}

	my ($key, $value);
	while (($key, $value) = each %$pVariableTable)# for every variable in the table
		{
		$sString =~ s/$key/$value/isg;				# replace the variable with its value
		}
	
	return ($::SUCCESS, '', $sString, 0);
	}

#######################################################
#																		
# ReturnToLastPage - bounce the browser to the previous
#	page
#
# Params:	0 - bounce delay (if less than 0, don't
#					automatically bounce)
#				1 - string to add to display
#				2 - optional page title.  If the page
#						title exists (ne ''), the page is formatted
#						using the bounce template
#				3 - pointer to the page list
#				4 - the refering site URL
#				5 - content site URL
#				6 - pointer to the setup blob
#				7+ - InputHash table
#
# Returns:	0 - status
#				1 - error message
#				2 - HTML for the bounce page
#
#######################################################

sub ReturnToLastPage
	{
#? ACTINIC::ASSERT($#_ > 7, "Invalid argument count in ReturnToLastPage ($#_)", __LINE__, __FILE__);
	
	if ($_[1] ne '')										# if the page title is defined, format the page prettily
		{
		return (ReturnToLastPageEnhanced(@_));
		}
	else														# otherwise, use a plain page
		{
		return (ReturnToLastPagePlain(@_));
		}
	}

#######################################################
#																		
# GroomError - make the error look nice for the HTML
#
# Params:	0 - Error string
#
# Returns:	0 - pretty string
#
#######################################################

sub GroomError
	{
	if ($#_ != 0)
		{
		return (GroomError(ACTINIC::GetPhrase(-1, 12, 'GroomError')));
		}
	my ($sError) = @_;

	if ($sError eq "")
		{
		return ($sError);
		}

	$sError = "<TABLE CELLPADDING=\"10\" WIDTH=\"550\" BORDER=\"1\" BGCOLOR=\"$$::g_pSetupBlob{FORM_BACKGROUND_COLOR}\">" .
		"<TR><TD><BIG> $sError</BIG></TD></TR></TABLE><P><HR>";

	return ($sError);
	}

#######################################################
#																		
# GroomHTML - Display HTML in catalog style
#	NOTE: this is a wrapper for the ACTINIC
#	package version.  It prevents a bunch of duplicate
#	work
#
# Params:	[0] - string to add to display
#				[1] - optional page title.  If the page
#						title exists, the page is formatted
#						using the bounce template
#				2 - pointer to the page list
#				3 - the refering site URL
#				4 - content site URL
#				5 - pointer to the setup blob
#				6+ - InputHash table
#
# Expects:	%::g_InputHash should be defined
#
# Returns:	($ReturnCode, $Error, $sHTML, 0)
#				if $ReturnCode = $::FAILURE, the operation failed
#					for the reason specified in $Error
#				Otherwise everything is OK
#				$sHTML - the HTML of the page
#
#######################################################

sub GroomHTML
	{
#? ACTINIC::ASSERT($#_ > 6, "Invalid argument count in GroomHTML ($#_)", __LINE__, __FILE__);

	my ($sHTML, $sMessage, $sRefPage, $sScriptName);
	my (%InputHash, $temp, $pPageList, $sTitle, $pSetupBlob, $sContentUrl, $sWebSiteUrl);
	($sMessage, $sTitle, $pPageList, $sWebSiteUrl, $sContentUrl, $pSetupBlob, %InputHash) = @_;
	
	pop @$pPageList;										# throw out the current page
	$sRefPage = pop @$pPageList;						# get the previous page
			
	$sScriptName = GetScriptNameRegexp();
	if ($sRefPage =~ /$sScriptName/)					# if the referring page was a script call,
		{
		#
		# get the page history - note that passing '' as the first argument guarantees tha prevquery will
		# be meaningless
		#
		my ($status, $sMessage, $sPrevQuery, $sPageHistory) = PrepareRefPageData('', $pPageList, $::TRUE);
		if ($status != $::SUCCESS)
			{
			return($status, $sMessage, '');
			}
		#
		# tack the "END" on so ReadAndParseInput knows this was a bounce
		#
		$sRefPage .= '&' . "REFPAGE=" . $sPageHistory . "END";		# this must be the last thing in the query statement
		}
	
	return (GroomHTMLEnhanced($sMessage, $sTitle, $pPageList,
		$sWebSiteUrl, $sContentUrl, $pSetupBlob, $sRefPage, \%InputHash));
	}

#######################################################
#																		
# GroomHTMLEnhanced - Format the page contents using
#	the bounce.html template
#
# Params:	0 - string to add to display
#				0 - optional page title.  If the page
#						title exists (ne ''), the page is formatted
#						using the bounce template
#				2 - pointer to the page list
#				3 - the refering site URL
#				4 - content site URL
#				5 - pointer to the setup blob
#				6 - the page to go to
#				7 - pointer to InputHash table
#
# Returns:	0 - status
#				1 - error message
#				2 - HTML for the page
#
#######################################################

sub GroomHTMLEnhanced
	{
#? ACTINIC::ASSERT($#_ > 6, "Invalid argument count in GroomHTMLEnhanced ($#_)", __LINE__, __FILE__);
	my ($sHTML, $sMessage, $sScriptName);
	my ($pInputHash, $temp, $pPageList, $sTitle, $pSetupBlob, $sWebSiteUrl, $sContentUrl, $sRefPage);
	($sMessage, $sTitle, $pPageList, $sWebSiteUrl, $sContentUrl, $pSetupBlob, , $sRefPage, $pInputHash) = @_;
	
	my ($sPath, @Response, $Status, $Message);
	$sPath = GetPath();									# get the path to the web site dir
	
	my (%VariableTable);
	$VariableTable{$::VARPREFIX."BOUNCETITLE"} = $sTitle; # add the title to the var list
	$VariableTable{$::VARPREFIX."BOUNCEMESSAGE"} = $sMessage; # add the message to the var list
	
	@Response = TemplateFile($sPath."bounce.html", \%VariableTable); # make the substitutions
	($Status, $Message, $sHTML) = @Response;
	if ($Status != $::SUCCESS)
		{
		return (@Response);
		}
	
	#######
	# make the file references point to the correct directory
	#######
	if( !$ACTINIC::B2B->Get('UserDigest') )
		{
		@Response = ACTINIC::MakeLinksAbsolute($sHTML, $::g_sWebSiteUrl, $::g_sContentUrl);
		}
	else
		{
		my $sBaseFile = $ACTINIC::B2B->Get('BaseFile');
		my $smPath = ($sBaseFile) ? $sBaseFile : $::g_sContentUrl;
		my $sCgiUrl = $::g_sAccountScript;
		$sCgiUrl   .= ($::g_InputHash{SHOP} ? '?SHOP=' . ACTINIC::EncodeText2($::g_InputHash{SHOP}, $::FALSE) . '&': '?');
		$sCgiUrl   .= 'PRODUCTPAGE=';
		@Response = ACTINIC::MakeLinksAbsolute($sHTML, $sCgiUrl, $smPath);
		}

	($Status, $Message, $sHTML) = @Response;
	if ($Status != $::SUCCESS)
		{
		return (@Response);
		}
	
	return ($::SUCCESS, '', $sHTML, 0);
	}

#######################################################
#																		
# ReturnToLastPagePlain - bounce the browser to the
#	previous page using a plain white page
#
# Params:	0 - bounce delay (if less than 0, don't
#					automatically bounce)
#				1 - string to add to display
#				2 - optional page title.  If the page
#						title exists (ne ''), the page is formatted
#						using the bounce template
#				3 - pointer to the page list
#				4 - the refering site URL
#				5 - content site URL
#				6 - pointer to the setup blob
#				7+ - InputHash table
#
# Returns:	0 - status
#				1 - error message
#				2 - HTML for the bounce page
#
#######################################################

sub ReturnToLastPagePlain
	{
#? ACTINIC::ASSERT($#_ > 7, "Invalid argument count in ReturnToLastPagePlain ($#_)", __LINE__, __FILE__);
	my ($sHTML, $nDelay, $sMessage, $sRefPage, $sScriptName, %InputHash, $temp, $pPageList, $sWebSiteUrl, $sContentUrl, $pSetupBlob);
	($nDelay, $sMessage, $temp, $pPageList, $sWebSiteUrl, $sContentUrl, $pSetupBlob, %InputHash) = @_;
	
	pop @$pPageList;										# throw out the current page
	$sRefPage = pop @$pPageList;						# get the previous page
	
	$sScriptName = GetScriptNameRegexp();
	if ($sRefPage =~ /$sScriptName/)					# if the referring page was a script call,
		{
		#
		# get the page history - note that passing '' as the first argument guarantees tha prevquery will
		# be meaningless
		#
		my ($status, $sMessage, $sPrevQuery, $sPageHistory) = PrepareRefPageData('', $pPageList, $::TRUE);
		if ($status != $::SUCCESS)
			{
			return($status, $sMessage, '');
			}
		#
		# tack the "END" on so ReadAndParseInput knows this was a bounce
		#
		$sRefPage .= '&' . "REFPAGE=" . $sPageHistory . "END";		# this must be the last thing in the query statement
		}
		
	return (BounceToPagePlain($nDelay, $sMessage, $temp, $pPageList,
		$sWebSiteUrl, $sContentUrl, $pSetupBlob, $sRefPage, \%InputHash));
	}

#######################################################
#																		
# BounceToPagePlain - bounce the browser to the
#	specified page using a simple page
#
# Params:	0 - bounce delay (if less than 0, don't
#					automatically bounce)
#				1 - string to add to display
#				2 - optional page title.  If the page
#						title exists (ne ''), the page is formatted
#						using the bounce template
#				3 - pointer to the page list
#				4 - the refering site URL
#				5 - content site URL
#				6 - pointer to the setup blob
#				7 - URL to go to
#				8 - InputHash table
#				9 - clear frames flag - if $::TRUE,
#				   clear any existing
#					frames when bouncing.   Default: $::FALSE
#
# Returns:	0 - status
#				1 - error message
#				2 - HTML for the bounce page
#
#######################################################

sub BounceToPagePlain
	{
#? ACTINIC::ASSERT($#_ > 7, "Wrong number of arguments in BounceToPagePlain ($#_)", __LINE__, __FILE__);

	my ($sHTML, $nDelay, $sMessage, $sRefPage, $sScriptName, $pInputHash);
	my ($temp, $pPageList, $sWebSiteUrl, $sContentUrl, $pSetupBlob, $bClearFrames);
	($nDelay, $sMessage, $temp, $pPageList, $sWebSiteUrl, $sContentUrl, $pSetupBlob, $sRefPage, $pInputHash, $bClearFrames) = @_;
	#
	# set the flag to clear the flag if it exists and clearing was requested
	#
#	$bClearFrames = $bClearFrames;					# What is this for?
			
	my ($sDigest,$sBaseFile) = ACTINIC::CaccGetCookies();
	if( !$sDigest )
		{
		$sWebSiteUrl = $sContentUrl;
		}
	else
		{
		$sWebSiteUrl = $sBaseFile;
		$sWebSiteUrl =~ s#/[^/]*$#/#;
		}
	if ($sRefPage eq '')									# if no referring page, ask the user to manually return
		{
		$sHTML = "<HTML>\n";								# open page
		$sHTML .= "<BODY";								# body definition
		if ($$pSetupBlob{'BACKGROUND_IS_IMAGE'} &&
			 length $$pSetupBlob{'BACKGROUND_VALUE'} > 0)
			{
			$sHTML .= " BACKGROUND=\"" . $sWebSiteUrl . $$pSetupBlob{'BACKGROUND_VALUE'} . "\"";
			}
		elsif (length $$pSetupBlob{'BACKGROUND_VALUE'} > 0)
			{
			$sHTML .= " BGCOLOR=\"" . $$pSetupBlob{'BACKGROUND_VALUE'} . "\"";
			}
		if (length $$pSetupBlob{'FOREGROUND_COLOR'} > 0)
			{
			$sHTML .= " TEXT=\"" . $$pSetupBlob{'FOREGROUND_COLOR'} . "\""
			}
		if (length $$pSetupBlob{'LINK_COLOR'} > 0)
			{
			$sHTML .= " LINK=\"" . $$pSetupBlob{'LINK_COLOR'} . "\""
			}
		if (length $$pSetupBlob{'ALINK_COLOR'} > 0)
			{
			$sHTML .= " ALINK=\"" . $$pSetupBlob{'ALINK_COLOR'} . "\""
			}
		if (length $$pSetupBlob{'VLINK_COLOR'} > 0)
			{
			$sHTML .= " VLINK=\"" . $$pSetupBlob{'VLINK_COLOR'} . "\""
			}
		$sHTML .= "><BLOCKQUOTE>\n";
		$sHTML .= $sMessage."<P>\n";					# add the call specific message (if any)
		$sHTML .= GetPhrase(-1, 22) . "<BR></BLOCKQUOTE>\n";
		}
	else														# bounce to the referring page
		{			
		$sHTML = "<HTML>\n";								# open page
		if( $sRefPage =~ /\?/ )
			{
			my $sBefore = "$`\?";
			my $sAfter = "\&$'";
			if( $$pInputHash{MAINFRAMEURL} )	# For parsed frameset we may change main frame URL
				{
				$sRefPage = $sBefore . 'MAINFRAMEURL=' . $$pInputHash{MAINFRAMEURL} . $sAfter;
				}
			elsif( $$pInputHash{BASE}  )			# Someone is passing catalog directory, pass it on
				{
				$sRefPage = $sBefore . 'BASE=' . $$pInputHash{BASE} . $sAfter;
				}
			}
		if ($nDelay >= 0)									# only try to auto bounce if the delay is a positive number
			{
			my $sMetaTag;
			if ($bClearFrames)							# use JavaScript to clear frames on the auto-bounce
				{
				$sMetaTag =
					"<SCRIPT LANGUAGE=\"JAVASCRIPT\">\n" . 
					"<!-- hide from older browsers\n" .
					"setTimeout(\"ForwardPage()\", " . 1000 * $nDelay . ");\n" .
					"function ForwardPage()\n" .
					"	{\n" .
					"	parent.location.replace('$sRefPage');\n" .
					"	}\n" .
					"// -->\n" .
					"</SCRIPT>\n";
				}
			else												# no need for the JavaScript, so use the more commonly supported Meta tag
				{
				$sMetaTag = "<META HTTP-EQUIV=\"refresh\" "; # refresh message
				$sMetaTag .= "CONTENT=\"$nDelay; URL=".$sRefPage."\">\n";
				}
			$sHTML .= $sMetaTag;
			}
		
		$sHTML .= "<BODY";								# body definition
		if ($$pSetupBlob{'BACKGROUND_IS_IMAGE'} &&
			 length $$pSetupBlob{'BACKGROUND_VALUE'} > 0)
			{
			$sHTML .= " BACKGROUND=\"" . $sWebSiteUrl . $$pSetupBlob{'BACKGROUND_VALUE'} . "\"";
			}
		elsif (length $$pSetupBlob{'BACKGROUND_VALUE'} > 0)
			{
			$sHTML .= " BGCOLOR=\"" . $$pSetupBlob{'BACKGROUND_VALUE'} . "\"";
			}
		if (length $$pSetupBlob{'FOREGROUND_COLOR'} > 0)
			{
			$sHTML .= " TEXT=\"" . $$pSetupBlob{'FOREGROUND_COLOR'} . "\""
			}
		if (length $$pSetupBlob{'LINK_COLOR'} > 0)
			{
			$sHTML .= " LINK=\"" . $$pSetupBlob{'LINK_COLOR'} . "\""
			}
		if (length $$pSetupBlob{'ALINK_COLOR'} > 0)
			{
			$sHTML .= " ALINK=\"" . $$pSetupBlob{'ALINK_COLOR'} . "\""
			}
		if (length $$pSetupBlob{'VLINK_COLOR'} > 0)
			{
			$sHTML .= " VLINK=\"" . $$pSetupBlob{'VLINK_COLOR'} . "\""
			}
		$sHTML .= "><BLOCKQUOTE>\n";
		$sHTML .= $sMessage."<P>\n";					# add the call specific message (if any)
		my $sBounceSentence;
		if ($nDelay >= 0)									# if the delay is a positive number
			{
			$sBounceSentence = GetPhrase(-1, 23, $sRefPage) . "\n"; # try to automatically bounce or here
			}
		else													# negative delay means no auto bounce
			{
			$sBounceSentence = GetPhrase(-1, 161, $sRefPage) . "\n"; # click here to continue
			}
		#
		# if we are to clear the frames in the jump, add the target to this URL
		#
		if ($bClearFrames)
			{
			$sBounceSentence =~ s/(HREF=)/TARGET="_parent" $1/i;
			}
		#
		# add the message to the page
		#
		$sHTML .= $sBounceSentence . "<BLOCKQUOTE>";
		}
	$sHTML .= "</BODY>\n</HTML>\n";
	
	return ($::SUCCESS, '', $sHTML, 0);
	}

#######################################################
#																		
# ReturnToLastPageEnhanced - bounce the browser to the
#	previous page, but format the page contents using
#	the bounce.html template
#
# Params:	0 - bounce delay (if less than 0, don't
#					automatically bounce)
#				1 - string to add to display
#				2 - optional page title.  If the page
#						title exists (ne ''), the page is formatted
#						using the bounce template
#				3 - pointer to the page list
#				4 - the refering site URL
#				5 - content site URL
#				6 - pointer to the setup blob
#				7+ - InputHash table
#
# Returns:	0 - status
#				1 - error message
#				2 - HTML for the bounce page
#
#######################################################

sub ReturnToLastPageEnhanced
	{
#? ACTINIC::ASSERT($#_ > 7, "Invalid argument count in ReturnToLastPageEnhanced ($#_)", __LINE__, __FILE__);
	my ($sHTML, $nDelay, $sMessage, $sRefPage, $sScriptName);
	my (%InputHash, $temp, $pPageList, $sTitle, $sMetaTag, $pSetupBlob, $sContentUrl, $sWebSiteUrl);
	($nDelay, $sMessage, $temp, $pPageList, $sWebSiteUrl, $sContentUrl, $pSetupBlob, %InputHash) = @_;
	
	pop @$pPageList;										# throw out the current page
	$sRefPage = pop @$pPageList;						# get the previous page

	$sScriptName = GetScriptNameRegexp();
	if ($sRefPage =~ /$sScriptName/)					# if the referring page was a script call,
		{
		#
		# get the page history - note that passing '' as the first argument guarantees tha prevquery will
		# be meaningless
		#
		my ($status, $sMessage, $sPrevQuery, $sPageHistory) = PrepareRefPageData('', $pPageList, $::TRUE);
		if ($status != $::SUCCESS)
			{
			return($status, $sMessage, '');
			}
		#
		# tack the "END" on so ReadAndParseInput knows this was a bounce
		#
		$sRefPage .= '&' . "REFPAGE=" . $sPageHistory . "END";		# this must be the last thing in the query statement
		}
	
	return (BounceToPageEnhanced($nDelay, $sMessage, $sTitle, $pPageList,
		$sWebSiteUrl, $sContentUrl, $pSetupBlob, $sRefPage, \%InputHash));
	}

#######################################################
#																		
# BounceToPageEnhanced - bounce the browser to the
#	specified page, but format the page contents using
#	the bounce.html template
#
# Params:	0 - bounce delay (if less than 0, don't
#					automatically bounce)
#				1 - string to add to display
#				2 - optional page title.  If the page
#						title exists (ne ''), the page is formatted
#						using the bounce template
#				3 - pointer to the page list
#				4 - the refering site URL
#				5 - content site URL
#				6 - pointer to the setup blob
#				7 - the page to go to
#				8 - pointer to InputHash table
#				9 - clear frames flag - if $::TRUE,
#					clear any existing
#					frames when bouncing.   Default: $::FALSE
#
# Returns:	0 - status
#				1 - error message
#				2 - HTML for the bounce page
#
#######################################################

sub BounceToPageEnhanced
	{
#? ACTINIC::ASSERT($#_ > 7, "Wrong number of arguments in BounceToPageEnhanced ($#_)", __LINE__, __FILE__);
	my ($sHTML, $nDelay, $sMessage, $sScriptName);
	my ($pInputHash, $temp, $pPageList, $sTitle, $sMetaTag, $pSetupBlob, $sWebSiteUrl, $sContentUrl, $sRefPage, $bClearFrames);
	($nDelay, $sMessage, $sTitle, $pPageList, $sWebSiteUrl, $sContentUrl, $pSetupBlob, , $sRefPage, $pInputHash, $bClearFrames) = @_;
	#
	# set the flag to clear the flag if it exists and clearing was requested
	#
	#	$bClearFrames = $bClearFrames;				# I don't see what this is for (rz)
		
	if ($sRefPage eq '')									# if no referring page, ask the user to manually return
		{
		$sMessage .= "<P>\n";							# add the bouncy message
		$sMessage .= GetPhrase(-1, 22) . "<BR>\n";
		$sMetaTag = '';									# no bounce command
		}
	else														# bounce to the referring page
		{	
		if( $sRefPage =~ /\?/ )
			{
			my $sBefore = "$`\?";
			my $sAfter = "\&$'";
			if( $$pInputHash{MAINFRAMEURL} )	# For parsed frameset we may change main frame URL
				{
				$sRefPage = $sBefore . 'MAINFRAMEURL=' . $$pInputHash{MAINFRAMEURL} . $sAfter;
				}
			elsif( $$pInputHash{BASE}  )			# Someone is passing catalog directory, pass it on
				{
				$sRefPage = $sBefore . 'BASE=' . $$pInputHash{BASE} . $sAfter;
				}
			}
		if ($nDelay >= 0)									# only try to auto bounce if the delay is a positive number
			{
			if ($bClearFrames)							# use JavaScript to clear frames on the auto-bounce
				{
				$sMetaTag =
					"<SCRIPT LANGUAGE=\"JAVASCRIPT\">\n" . 
					"<!-- hide from older browsers\n" .
					"setTimeout(\"ForwardPage()\", " . 1000 * $nDelay . ");\n" .
					"function ForwardPage()\n" .
					"	{\n" .
					"	parent.location.replace('$sRefPage');\n" .
					"	}\n" .
					"// -->\n" .
					"</SCRIPT>\n";
				}
			else												# no need for the JavaScript, so use the more commonly supported Meta tag
				{
				$sMetaTag = "<META HTTP-EQUIV=\"refresh\" "; # refresh message
				$sMetaTag .= "CONTENT=\"$nDelay; URL=".$sRefPage."\">\n";
				}
			}
		
		$sMessage .= "<P>\n";							# add the bouncy message
		my $sBounceSentence;
		if ($nDelay >= 0)									# if the delay is a positive number
			{
			$sBounceSentence = GetPhrase(-1, 23, $sRefPage) . "\n"; # try to automatically bounce or here
			}
		else													# negative delay means no auto bounce
			{
			$sBounceSentence = GetPhrase(-1, 161, $sRefPage) . "\n"; # click here to continue
			}
		#
		# if we are to clear the frames in the jump, add the target to this URL
		#
		if ($bClearFrames)
			{
			$sBounceSentence =~ s/(HREF=)/TARGET="_parent" $1/i;
			}
			
		$sMessage .= $sBounceSentence;				# add the bounce line to the text
		}
	
	my ($sPath, @Response, $Status, $Message);
	$sPath = GetPath();									# get the path to the web site dir

	my (%VariableTable);
	$VariableTable{$::VARPREFIX."BOUNCETITLE"} = $sTitle; # add the title to the var list
	$VariableTable{$::VARPREFIX."BOUNCEMESSAGE"} = $sMessage; # add the message to the var list
	
	@Response = TemplateFile($sPath."bounce.html", \%VariableTable); # make the substitutions
	($Status, $Message, $sHTML) = @Response;
	if ($Status != $::SUCCESS)
		{
		return (@Response);
		}
	
	#######
	# make the file references point to the correct directory
	#######
	my $smPath = $sContentUrl;
	my $sCgiUrl = $sWebSiteUrl;
	my ($sDigest,$sBaseFile) = ACTINIC::CaccGetCookies();
	if( $sDigest )
		{
		$smPath = ($sBaseFile) ? $sBaseFile : $sContentUrl;
		$sCgiUrl = $::g_sAccountScript;
		$sCgiUrl   .= $::g_InputHash{SHOP} ? '?SHOP=' . ACTINIC::EncodeText2($::g_InputHash{SHOP}, $::FALSE) . '&' : '?';
		$sCgiUrl   .= 'PRODUCTPAGE=' . $sRefPage;
		}
	@Response = MakeLinksAbsolute($sHTML, $sCgiUrl, $smPath);
	($Status, $Message, $sHTML) = @Response;
	if ($Status != $::SUCCESS)
		{
		return (@Response);
		}
	
	my ($sSearchTag, $sReplaceTag);
	$sSearchTag = '</TITLE>';							# the bounce meta tag comes immediately after the title
	$sReplaceTag = $sSearchTag . "\n" . $sMetaTag;
	
	$sHTML =~ s/$sSearchTag/$sReplaceTag/ig;		# insert the bounce meta tag
	
	return ($::SUCCESS, '', $sHTML, 0);
	}

#######################################################
#																		
# UpdateDisplay - Print the HTML to the browser after
#	modifying it to keep the page refs in order
#
# Params:	0 - HTML
#				1 - the original CGI input string
#				2 - pointer to the page list
#			   3 - Cookie (optional)
#				4 - cache flag (optional - default no-cache)
#				5 - contact details cookie (optional)
#
#######################################################

sub UpdateDisplay
	{
#? ACTINIC::ASSERT($#_ >= 2, "Invalid argument count in UpdateDisplay ($#_)", __LINE__, __FILE__);
	my ($sHTML, $OriginalInputData, $pPageList, $sCookie, $bNoCacheFlag, $sContactDetailsCookie) = @_;
	if (!defined $sCookie)								# if the optional cookie was not supplied
		{
		$sCookie = '';										# set the cookie to empty
		}
	if (!defined $bNoCacheFlag)						# default the cache flag to no cache
		{
		$bNoCacheFlag = $::TRUE;
		}
	
	###
	# supply the page list
	###
	my ($sSearch, $sReplace, $sPrefQuery);
	$sSearch = $::VARPREFIX."REFPAGE";
	
	my ($status, $sMessage, $sPrevQuery, $sPageHistory) = PrepareRefPageData($OriginalInputData, $pPageList, $::FALSE);
	if ($status != $::SUCCESS)
		{
		TerminalError($sMessage);
		}
	$sPageHistory =~ s/\|\|\|$//;						# strip the trailing terminator
	$sReplace = "<INPUT TYPE=HIDDEN NAME=REFPAGE VALUE=\"$sPageHistory\">\n" .
	"<INPUT TYPE=HIDDEN NAME=PREVQUERY VALUE=\"$sPrevQuery\">\n"; # add the query string in case it
	# gets lost (some servers don't include it in the HTTP_REFERER)
	$sHTML =~ s/$sSearch/$sReplace/;					# insert the page list

	#
	# add a random hidden parameter value to guarantee requeries
	#
	srand();
	my ($Random) = rand();
	$sHTML =~ s/NETQUOTEVAR:RANDOM/$Random/g;
	
	PrintPage($sHTML, $sCookie, $bNoCacheFlag, $sContactDetailsCookie);
	}

#######################################################
#
# PrintNonParsedHeader - print the non-parsed headers
#  Note that this function is separate from PrintHeader
#  because I didn't want to break access to PrintHeader
#  at such a late date.  This function should be called
#  when dynamic feedback is required.  Note that NT does
#  not respect nonparsed headers for dynamic update (even
#  under Apache).
#
#	Input: 	0 - content type
#
#######################################################

sub PrintNonParsedHeader
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in PrintNonParsedHeader ($#_)", __LINE__, __FILE__);
	# 
   # Dump the HTTP headers so we can do proper non parsed header processing (for dynamic feedback)
   #
	$|=1;
	print $::ENV{SERVER_PROTOCOL} . " 200 OK\n";
	print "Server: " . $::ENV{SERVER_SOFTWARE} . "\n";
	print "Content-type: " . $_[0] . "\n";
   #
   # Build a date for the expiry
   #
	my ($day, $month, $now, $later, $expiry, @now, $sNow);
	my (@days) = ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat');
	my (@months) = ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');
	
	$now = time;
	@now = gmtime($now);
	$day = $days[$now[6]];
	$month = $months[$now[4]];
	$sNow = sprintf("%s, %02d %s %04d %02d:%02d:%02d GMT", $day, $now[3],
						 $month, $now[5]+1900, $now[2], $now[1], $now[0]);

	print "Date: $sNow\n\n";							# print the date to allow the browser to compensate between server and client differences
	}

#######################################################
#
# PrintHeader - print the HTTP header
#
#	Params: 	0 - content type
#				1 - content length
#				2 - cookie if any (or undef)
#				3 - no-cache flag - if $::TRUE,
#					include no-cache flag.
#				4 - contact details cookie (optional)
#
# 3/11/99 - content type, length, date and nocache moved to the top
#		date made unconditional.	R. Zybert
#
#######################################################

sub PrintHeader
	{
#? ACTINIC::ASSERT($#_ >= 3, "Invalid argument count in PrintHeader ($#_)", __LINE__, __FILE__);

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sType, $nLength, $sCookie, $bNoCache, $sContactDetailsCookie) = @_;
	#
	# Turn on non-parsed headers by default when running under IIS server and Doug MacEachern's modperl
	#
	my $bNPH = $::FALSE;
	if ( (defined($ENV{'SERVER_SOFTWARE'}) && $ENV{'SERVER_SOFTWARE'}=~/IIS/) ||
		  (defined($ENV{'GATEWAY_INTERFACE'}) && $ENV{'GATEWAY_INTERFACE'} =~ /^CGI-Perl/))
		{
		$bNPH = $::TRUE;
		}

	#
	# Build a date for the expiry
	#
	my (@expires, $day, $month, $now, $later, $expiry, @now, $sNow);
	my (@days) = ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat');
	my (@months) = ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');
	
	$now = time;
	@now = gmtime($now);
	$day = $days[$now[6]];
	$month = $months[$now[4]];
	$sNow = sprintf("%s, %02d %s %04d %02d:%02d:%02d GMT", $day, $now[3],
							$month, $now[5]+1900, $now[2], $now[1], $now[0]);
	$later = $now + 2 * 365 * 24 * 3600;			# Time in 2 years
	@expires = gmtime($later);							# grab time components
	$day = $days[$expires[6]];
	$month = $months[$expires[4]];
	$expiry = sprintf("%s, %02d-%s-%04d %02d:%02d:%02d GMT", $day, $expires[3],
							$month, $expires[5]+1900, $expires[2], $expires[1], $expires[0]);
	#
	# set the cookie if it needs to be set
	#
	my ($sCurrentCookie);
	if (!$ACTINIC::AssertIsActive)
		{
		$sCurrentCookie = GetCookie();
		}
	my $bCookie = ( (length $sCookie) > 0 &&		# if a cookie is to be saved
			$sCurrentCookie ne $sCookie);				# and it is a new value
	#
	# now print the header
	#
	if ($bNPH)
		{
		print "HTTP/1.0 200 OK\n";						# the status
		}
	
	print "Content-type: $sType\n";
	print "Content-length: $nLength\n";
	print "Date: $sNow\n";							# print the date to allow the browser to compensate between server and client differences

	if ($bNoCache)
		{
		print "Pragma: no-cache\n";
		}

	if ($bCookie)											# if we are to save the cookie
		{
		print "Set-Cookie: ACTINIC_CART=" .			# set the cookie
		   $sCookie . "; EXPIRES=" .
			$expiry . "; PATH=/;\n";
		}
	
	if (!$ACTINIC::AssertIsActive)
		{
		my $sBusinessCookie = ACTINIC::CAccBusinessCookie();			# If B2B user logged in - save the digest
		if ($sBusinessCookie eq "-" and $sContactDetailsCookie)		# if we are to save the contact details cookie
			{
			print "Set-Cookie: " . $sContactDetailsCookie . # set the cookie
				"; EXPIRES=" . $expiry . "; PATH=/;\n";
			}
		else
			{
			print "Set-Cookie: ACTINIC_BUSINESS=" . $sBusinessCookie . 		# set the cookie - this session only
				"; PATH=/;\n";
			}
		if ($::ACT_ADB)								# If there is an address book
			{
			print $::ACT_ADB->Header();			# Ouput address book cookies
			}
		}

	print "\n";
	}

#######################################################
#
# PrintPage - print the HTML page
#
#	Params: 	0 - HTML to print
#				1 - cookie if any (or undef)
#				2 - no-cache flag - if $::TRUE,
#					include no-cache flag.
#					Default - $::TRUE
#				3 - contact details cookie (optional)
#
#######################################################

sub PrintPage
	{
#? ACTINIC::ASSERT($#_ >= 1, "Invalid argument count in PrintPage ($#_)", __LINE__, __FILE__);
   if ($::s_nErrorRecursionCounter > 10)
		{
		$ACTINIC::AssertIsActive = $::TRUE;
#?      ACTINIC::TRACE('Callstack:\n%s', CallStack());
		}
   $::s_nErrorRecursionCounter++;

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($nLength, $sHTML, $sCookie, $bNoCacheFlag, $sContactDetailsCookie);
	($sHTML, $sCookie, $bNoCacheFlag, $sContactDetailsCookie) = @_;
	
   if (!$ACTINIC::AssertIsActive)					 # skip the XML parsing if we are reporting an assert as this can cause infinite looping if the problem is in the customer account code
		{
		$sHTML = ACTINIC::ParseXML($sHTML);			# the body
		}
	$nLength = length $sHTML;
	
	if (!defined $bNoCacheFlag)						# default the no cache flag to on
		{
		$bNoCacheFlag = $::TRUE;
		}
	
	PrintHeader('text/html', $nLength, $sCookie, $bNoCacheFlag, $sContactDetailsCookie);

	binmode STDOUT;										# dump in binary mode since Netscape likes it
	
	print $sHTML;							# the body
	}

#######################################################
#
# PrintText - print the text page
#
#	Params: 	0 - text to print
#
#######################################################

sub PrintText
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in PrintText ($#_)", __LINE__, __FILE__);

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my $sText = $_[0];
	
	my $nLength = length $sText;
	
	PrintHeader('text/plain', $nLength, undef, $::FALSE);

	binmode STDOUT;										# dump in binary mode since Netscape likes it

	print $sText;											# the body
	}

#######################################################
#																		
# ReportError - report the specified error to the
#	browser and error file
#
# Params:	0 - error message
#				1 - the file path
#
#######################################################

sub ReportError
	{
#? ACTINIC::ASSERT($#_ == 1, "Invalid argument count in ReportError ($#_)", __LINE__, __FILE__);

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sMessage, $sPath);
	($sMessage, $sPath) = @_;
	
	RecordErrors(@_);										# record the error to the error file
	
	TerminalError($_[0]);								# display the error
	}

#######################################################
#																		
# RecordErrors - Record the specified error to the
#	error file
#
# Params:	0 - error message
#				1 - file path
#
#######################################################

sub RecordErrors
	{
#? ACTINIC::ASSERT($#_ == 1, "Invalid argument count in RecordErrors ($#_)", __LINE__, __FILE__);

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sMessage, $sPath);
	($sMessage, $sPath) = @_;
	
	#########
	# Write the error to the file
	#########
	my ($sPad, $sFormat, $sFile);
	$sPad = " "x100;
	$sFile = $sPath."error.err";
	
	SecurePath($sFile);									# make sure only valid filename characters exist in $file to prevent hanky panky
	
	open(NQFILE, ">>".$sFile);							# Open the error file
	
	print NQFILE ("Program = ");						# Begin to write error file details
	print NQFILE (substr($::prog_name.$sPad,0,8)); # Write error file details
	
	print NQFILE (", Program version = ");			# Write error file details
	print NQFILE (substr($::prog_ver.$sPad,0,6)); # Write error file details
	
	print NQFILE (", HTTP Server = ");				# Write error file details
	print NQFILE (substr($::ENV{'SERVER_SOFTWARE'}.$sPad,0,30)); # Write error file details
	
	print NQFILE (", Return code = ");				# Write error file details
	print NQFILE (substr("999".$sPad,0,20));		# Write error file details
	
	print NQFILE (", Date and Time = ");			# Write error file details
	
	my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst);
	($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime(time);	# platform independent time
	$mon++;													# make month 1 based
	$year += 1900;											# make year AD based
	$sFormat = sprintf("%2.2d/%2.2d/%4.4d %2.2d:%2.2d:%2.2d", $mday, $mon, $year, $hour, $min, $sec);
	print NQFILE ($sFormat);							# Write error file details
	
	print NQFILE (", Internal Errors = ");			# Write error file details
	print NQFILE ($sMessage);							# Write error file details
	
	print NQFILE "\n";
	close NQFILE;
	
	ChangeAccess("rw", $sFile);						# make the file accessible
	}

#######################################################
#
# TerminalError - generate the error html
#
#	Params: 	0 - the error
#
#######################################################

sub TerminalError
	{
# No assert here because the assert function calls this function - recursion loop
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sError, $sHTML);
	($sError) = @_;										# get the error message
	
	$sHTML  = "<HTML><TITLE>Actinic</TITLE><BODY>";
	if (defined $::g_pPromptList)
		{
		$sHTML .= "<H1>" . GetPhrase(-1, 24) . "</H1>";
		$sHTML .= "<HR>" . GetPhrase(-1, 25) . ": $sError<HR>";
		$sHTML .= GetPhrase(-1, 26);
		}
	else														# if the localized text file has not been read - assume english
		{
		$sHTML .= "<H1>" . "A General Script Error Occurred" . "</H1>";
		$sHTML .= "<HR>" . "Error" . ": $sError<HR>";
		$sHTML .= "Press the Browser back button and try again or contact your ISP.";
		}
	$sHTML .= "</BODY></HTML>";
	
	$ACTINIC::AssertIsActive = $::TRUE;
	PrintPage($sHTML, undef, $::TRUE);
	
	exit;
	}

#######################################################
#																		
# MakeLinksAbsolute - make all file references
#	absolute (to the web site dir)
#
# Params:	0 - current HTML
#				1 - referring site url
#				2 - content url
#
# Returns:	0 - status
#				1 - error message
#				2 - modified text
#
# 3/11/99 - modified to accept single quotes - R. Zybert
#
#######################################################

sub MakeLinksAbsolute
	{
#? ACTINIC::ASSERT($#_ == 2, "Invalid argument count in MakeLinksAbsolute ($#_)", __LINE__, __FILE__);
	
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	my ($sHTML, $sWebSiteUrl, $sContentUrl, $Status, $Message, @Response);
	($sHTML, $sWebSiteUrl, $sContentUrl) = @_;

	$sContentUrl =~ s#/[^/]*$#/#;
	
	#######
	# make the file references point to the correct directory
	# Absolute addresses (starting from /) are unchanged (rz)
	#######
	$sHTML =~ s/<IMG([^>]*?)SRC=(['"])?(?!http(s?):)([^'"\/][^"\s]+)(['"\s])/<IMG$1SRC=$2$sContentUrl$3$4$5/gi;	# '<emacs format> # replace image file references
	$sHTML =~ s/<BODY([^>]*?)BACKGROUND=(['"])?(?!http(s?):)([^'"\/][^'"\s]+)(['"\s])/<BODY$1BACKGROUND=$2$sContentUrl$3$4$5/gi;	# ' <quote helps emacs format> # replace background imagefile refs
	$sHTML =~ s/CODEBASE=(['"])?(?!http(s?):)([^'"\/][^"\s]+)(['"\s])/CODEBASE=$1$sContentUrl$2$3$4/gi;	# ' <quote helps emacs format> # replace codebase references
	$sHTML =~ s/\.src\s*=\s*(['"])(?!http(s?):)([^'"\/][^"'\s]+)(["'])/\.src = $1$sContentUrl$2$3$4/gi;	# ' <quote helps emacs format> # replace javascript images
  	$sHTML =~ s/<A([^>]*?)HREF=(['"])?(?!http(s?):|mailto:|#|\/|javascript:)([^'"\s]+)(['"\s])/<A$1HREF=$2$sWebSiteUrl$3$4$5/gi;	# " <quote helps emacs format> # replace hyperlink references
 	$sHTML =~ s/<FRAME([^>]*?)SRC=(['"])?(?!http(s?):|mailto:|#)([^'"\/][^'"\s]+)(["\s])/<FRAME$1SRC=$2$sWebSiteUrl$3$4$5/gi;	# " <quote helps emacs format> # replace hyperlink references
# 	$sHTML =~ s/<FRAME([^>]*?)SRC=(['"])?([^'"][^'"]+)(["'])/<FRAME$1SRC=$2$sWebSiteUrl$2$3$4/gi;	# Simpler frame format
	$sHTML =~ s/<INPUT([^>]*?)SRC=(['"])?(?!http(s?):)([^'"\/][^'"\s]+)(["\s])/<INPUT$1SRC=$2$sContentUrl$3$4$5/gi;	# " <quote helps emacs format> # replace image file references
	return ($::SUCCESS, '', $sHTML);					# do the replacement
	}
##################################################################################
#																											#
# HTML manipulation functions - end																#
#																											#
##################################################################################

##################################################################################
#																											#
# Generic Utilities - begin        																#
#																											#
##################################################################################

#######################################################
#
# GetScriptNameRegexp
#
# Returns: 	0 - a regexp that will match any of the
#					standard Catalog scriptnames
#
#######################################################

sub GetScriptNameRegexp
	{
	my (@ScriptPathParts) = split /(\\|\/)/, $::ENV{"SCRIPT_NAME"};
	my ($sScriptBase);
	$sScriptBase = substr($ScriptPathParts[$#ScriptPathParts], 2);
	return ("(ca|os|nq|ts|cp|ss|sh|bb|md)$sScriptBase");
	}
																							  
############################################################
#  IsStaticPage
#  Test URL to guess if it represents a static page
#  
#   Argument : URL
#   Result   : $::TRUE for static page
#  			   $::FALSE if not (or don't know)
#
#  Ryszard Zybert  Jul 24 20:32:07 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################
sub IsStaticPage
	{
	my ($sURL) = @_;
	my $sRegExp = GetScriptNameRegexp();
	if( $sURL =~ /(\.htm(l?)(\#[^\#]*)*)|(\/)$/i and $sURL !~ /$sRegExp/ )
		{
		return ($::TRUE);
		}
	return ($::FALSE);
	}
#######################################################
#
# Modulus - use this division function in place of
#	the % operator in cases where performance is not an
#	issue *or* when it is likely that the number is
#	greater than 2^31.  This is required because Perl
#	5.003 on FreeBSD crashes with a floating point exception
#	in those cases.
#
# Params:	0 - a
#				1 - b
#		where c = a % b
#
# Returns: 	0 - c
#
#######################################################

sub Modulus
	{
#? ACTINIC::ASSERT($#_ == 1, "Wrong number of arguments in Modulus ($#_)", __LINE__, __FILE__);
	my ($nA, $nB) = @_;
	#
	# a % b = int(a - b * int(a/b) )
	#
	my $nC = $nA - $nB * int($nA / $nB);
#?	if ($^O ne 'freebsd')
#?		{
#?		my $nD = $nA % $nB;
#? ACTINIC::ASSERT($nD == $nC, "Modulus emulation error $nC != $nD", __LINE__, __FILE__);
#?		}
	return($nC);
	}

#######################################################
#
# ReadTheDir
#     Open a directory and read its contents - this
#     is a hack-around for a bug in PerlIS for NT.
#
# Params: 	0 - the directory path to read
#
# Returns: 	0 - status code
#				1 - error message if any
#				2+ - file list (or 0, 0)
#
#######################################################

sub ReadTheDir
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadTheDir ($#_)", __LINE__, __FILE__);
	
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($sPath, @FileList);
	($sPath) = @_;											# get the path
	
	SecurePath($sPath);									# make sure only valid filename characters exist in $file to prevent hanky panky
	if( opendir (NQDIR, "$sPath") )					# open the directory to get a file listing
		{														# if successful,
		@FileList = readdir (NQDIR);					# read the directory
		closedir (NQDIR);									# close the directory
		return ($::SUCCESS, '', @FileList);			# return the directory contents
		}
	
	if ($^O ne "MSWin32")
		{
		return($::FAILURE, GetPhrase(-1, 31, $sPath, $!), 0, 0);
		}
	#
	# if we are here, the open failed.  This is probably NT with the PerliS 303 bug
	#	try to read the directory using dos commands
	#
	my ($sDosPath, $sCommand);
	$sDosPath = $sPath;									# get the path of the directory to read
	$sDosPath =~ s/\//\\/g;								# convert the forward slashes to dos backslashes
	
	$sCommand = "dir /B \"$sDosPath\"";
	
	unless (open (PIPE, $sCommand . " |"))
		{
		return($::FAILURE, GetPhrase(-1, 32, $sPath, $!), 0, 0);
		}
	
	@FileList = <PIPE>;									# read the contents of the directory
	chomp @FileList;										# remove the trailing newlines
	close (PIPE);											# close the file
	
	if ($#FileList == 0 &&								# if the command returned file not found
		 $FileList[0] =~ m/File Not Found/i)
		{
		my ($sMessage);
		$sMessage = $FileList[0];
		return($::FAILURE, GetPhrase(-1, 32, $sPath, $sMessage), 0, 0);
		}
	
	return ($::SUCCESS, '', @FileList);				# return the directory contents
	}

#######################################################
#																		
# IsCatalogFramed - Is Catalog running in framed mode
#
# Returns:	($ReturnCode)
#				$::TRUE if running in a Frame
#				$::FALSE if not
#
#######################################################

sub IsCatalogFramed
	{
	#
	# use the existence of navigation page 
	#
	return(CheckFileExists("framenavbar.html", GetPath())); 
	}

#######################################################
#																		
# CheckFileExists - returns whether the given file 
#				exists and is readable
#
# Params:	[0] - File name
#				[1] - Path
#
# Returns:	($ReturnCode)
#				$::TRUE if file exists and is readable
#				$::FALSE if not 
#
#######################################################

sub CheckFileExists
	{
#? ACTINIC::ASSERT($#_ == 1, "Wrong number of arguments in CheckFileExists", __LINE__, __FILE__);

	my ($sFileName, $sPath);
	($sFileName, $sPath) = @_;
	#
	# build the file name 
	#
	my $sFile = $sPath . $sFileName;
	return (-e $sFile && -r $sFile);					# does the file exist and is readable
	}

#######################################################
#																		
# GetCatalogBasePageName - gets the file name of the
#		enclosing frame
#
# Params:	[0] - Path
#
# Returns:	($ReturnCode, $sError, $sPageName)
#				$::TRUE if file exists and is readable, $::FALSE if not 
#				$sError if present or ""
#				$sBasePageName - base page name
#
#######################################################

sub GetCatalogBasePageName
	{
#? ACTINIC::ASSERT($#_ == 0, "Wrong number of arguments in GetCatalogBasePageName", __LINE__, __FILE__);

	my ($sPath, $sBasePageName, $sNavFileName);
	($sPath) = @_;
	#
	# build the file name 
	#
	my $sFile = "framenavbar.html";
	if(!CheckFileExists($sFile, $sPath))
		{
		return($::FALSE, "$sFile could not be found", "");
		}
	$sNavFileName = $sPath . $sFile;
	#
	# open the file
	#
	unless (open (NAVFILE, "<$sNavFileName"))
		{
		return ($::FALSE, ACTINIC::GetPhrase(-1, 21, $sNavFileName, $!), '');
		}
	#
	# find a HTML fragment with the base page name
	#
		{
		local $/ = undef;
		$_ = <NAVFILE>;									# read the entire file into $_

		($sBasePageName) = /\&BPN=([a-zA-Z0-9_.]+)[^>]*?TARGET="_(top|parent)"/i;
#? ACTINIC::ASSERT((length $sBasePageName) > 0, "Base page name not found", __LINE__, __FILE__);
		}
	close(NAVFILE);
	return ($::SUCCESS, "", $sBasePageName);		# return our base page name
	}

##################################################################################
#																											#
# Generic Utilities - end        																#
#																											#
##################################################################################

##############################################################################################################
#																		
# CGI Input Processing (should use CGI.pm but forbidden)- Begin
#
##############################################################################################################

#######################################################
#																		
# ReadAndParseInput - read the input and parse it
#
# Expects:	$::ENV to be defined
#
# Returns:	0 - status
#				1 - error message
#				2 - the input string
#				3 - spacer to keep output even
#				4+ - input hash table
#
#######################################################

sub ReadAndParseInput
	{
	my ($InputData, $nInputLength);
	
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	if ( (length $::ENV{'QUERY_STRING'}) > 0)		# if there is query string data (GET)
		{
		$InputData = $::ENV{'QUERY_STRING'};		# read it
		$nInputLength = length $InputData;
		}
	else														# otherwise, there must be a POST
		{
		my ($nStep, $InputBuffer);
		$nInputLength = 0;
		$nStep = 0;
		while ($nInputLength != $ENV{'CONTENT_LENGTH'})	# read until you have the entire chunk of data
			{
			#
			# read the input
			#
			binmode STDIN;
			$nStep = read(STDIN, $InputBuffer, $ENV{'CONTENT_LENGTH'});  # Set $::g_InputData equal to user input
			$nInputLength += $nStep;					# keep track of the total data length
			$InputData .= $InputBuffer;				# append the latest chunk to the total data buffer
			if (0 == $nStep)								# EOF
				{
				last;											# stop read
				}
			}
			
		if ($nInputLength != $ENV{'CONTENT_LENGTH'})
			{
			return ($::FAILURE, "Bad input.  The data length actually read ($nInputLength) does not match the length specified " . $ENV{'CONTENT_LENGTH'} . "\n", '', '', 0, 0);
			}	
		}
	$InputData =~ s/&$//;								# loose any bogus trailing &'s
	$InputData =~ s/=$/= /;								# make sure trailing ='s have a value
	my ($OriginalInputData);
	$OriginalInputData = $InputData;					# copy the input string for use later
	
	if ($nInputLength == 0)								# error if there was no input
		{
		return ($::FAILURE, "The input is NULL", '', '', 0, 0);
		}
	#
	# parse and decode the input
	#
	my (@CheckData, %DecodedInput);
	@CheckData = split (/[&=]/, $InputData);		# check the input line
	if ($#CheckData % 2 != 1)
		{
		return ($::FAILURE, "Bad input string \"" . $InputData . "\".  Argument count " . $#CheckData . ".\n", '', '', 0, 0);
		}
	my %EncodedInput = split(/[&=]/, $InputData);	# parse the input hash
	my ($key, $value);
	while (($key, $value) = each %EncodedInput)
		{
		$key = DecodeText($key, $ACTINIC::FORM_URL_ENCODED);	# decode the hash entry
		$value = DecodeText($value, $ACTINIC::FORM_URL_ENCODED);
		if ($key !~ /BLOB/i &&							# if the input is not an order blob
			 ($key =~ /\0/ ||								# check for poison NULLs
			  $value =~ /\0/))
			{
			return ($::FAILURE, "Input contains invalid characters.", undef, undef, undef, undef);
			}

		$DecodedInput{$key} = $value;
		}
	#
	# Now process the path to the catalog directory.  In stand alone mode, the path is hard coded in the script.
	# In Actinic Host mode, the path is derived from the SHOPID and the shop data file.
	#
	my ($status, $sError) = ProcessPath($DecodedInput{SHOP}, \%DecodedInput);
	if ($status != $::SUCCESS)
		{
		return ($status, $sError);
		}

	return ($::SUCCESS, '', $OriginalInputData, '', %DecodedInput);
	}

#######################################################
#																		
# ProcessPath - process the input to derive a path
#   to the catalog directory
#
# Params:	0 - shop ID if in Actinic Host Mode
#               or undef if stand alone
#
# Returns:	0 - status
#				1 - error message
#
#######################################################

sub ProcessPath
	{
#? ACTINIC::ASSERT($#_ >= 0, "Invalid argument count in ProcessPath ($#_)", __LINE__, __FILE__);
	my ($sShopID, $rhInput) = @_;
	my ($status, $sError);
	#
	# Now process the path to the catalog directory.  In stand alone mode, the path is hard coded in the script.
	# In Actinic Host mode, the path is derived from the SHOPID and the shop data file.
	#
	my $sInitialPath = 'NETQUOTEVAR:PATH';
	if (!NETQUOTEVAR:ACTINICHOSTMODE)				# stand alone mode
		{
		$ACTINIC::s_sPath = $sInitialPath;
		}
	else
		{
		#
		# Check if the shop ID has nothing in it
		#
		if ($sShopID eq '' && 
			($$rhInput{ACTION} eq 'AUTHORIZE' || $$rhInput{ACTION} eq 'OCC_VALIDATE'))
			{
			if(defined $$rhInput{PATH} && $$rhInput{PATH} ne '')
				{
				$ACTINIC::s_sPath = $$rhInput{PATH};
				return ($::SUCCESS, undef);
				}
			}
		#
		# Load the module for access to the configuration files
		#
		eval 'require MallUtil;';
		if ($@)												# the interface module does not exist
			{
			return ($::FAILURE, 'An error occurred loading the MallUtil module.  ' . $@);
			}
		#
		# Retrieve the appropriate record
		#
		my $pShop;
		($status, $sError) = MallUtil::GetShopRecordFromShopID($sShopID, \$pShop);
		if ($status != $::SUCCESS)
			{
			return ($status, $sError);
			}
		# 
		# Retrieve the specific path
		# 
		$ACTINIC::s_sPath = $pShop->{PATH};
		}

	return ($::SUCCESS, undef);
	}

#######################################################
#																		
# ProcessReferencePageData - keep track of the
#	reference page data
#
# Params:	0+ - InputHash
#
# Returns:	0 - status
#				1 - error message
#				2+ - page list
#
#######################################################

sub ProcessReferencePageData
	{
#? ACTINIC::ASSERT($#_ > 0, "Invalid argument count in ProcessReferencePageData ($#_)", __LINE__, __FILE__);
	
	my (%InputHash);
	(%InputHash) = @_;
	
	my ($sPages, @PageList);
	$sPages = $InputHash{"REFPAGE"};					# read the pagelist from the params
	if (defined $sPages)
		{
		@PageList = split (/\|\|\|/, $sPages);		# parse the list
		}
	else
		{
		@PageList = ();
		}
	
#	if ($#PageList != -1 &&								# if there are any entries and
	if ($#PageList > 0 &&								# if there are any entries (not just END) and
		 $PageList[$#PageList] eq "END")				# this was a page bounce
		{
		pop @PageList;										# drop the terminating "END" and don't add the bounce page to the
		}														# history list
	else
		{
		my ($sRefPage);
		$sRefPage = GetReferrer();
		($sRefPage) = split (/\&REFPAGE/, $sRefPage); # drop any refpage information from the referring page - we track that
		
		my ($sTopTag);
		$sTopTag = '#top';								# the top flag sometimes causes problems, so make sure it
		$sRefPage =~ s/$sTopTag//g;					# is stripped
		#
		# correct the referring page if we are using frames to make this CGI call and the call came
		# from the navigation bar.
		#
		if ($InputHash{BPN} ne '')						# this call was made from the navigation frame
			{
			my $nIndex;
			#
			# strip the filename from the static page URL
			#
			while ($sRefPage =~ /\//g)
				{
				$nIndex = pos $sRefPage;				# locate the last "/"
				}
			$sRefPage = substr ($sRefPage, 0, $nIndex);	# snag the url
			#
			# cat on the base page filename
			#
			$sRefPage .= $InputHash{BPN};
			}
		
		push (@PageList, $sRefPage);					# add the last page to the list
		}
	
	#####
	# make sure any CGI queries in the ref page list include their query_string
	#####
	my ($sScriptName) = GetScriptNameRegexp();
	if ($PageList[$#PageList] =~ /$sScriptName$/)# if the refpage indicates there is a script in the list that does
		{														# not have its query string,
		$PageList[$#PageList] .= "?" . $InputHash{'PREVQUERY'};	  	# add the query string
		}
	
	return ($::SUCCESS, '', @PageList);
	}

#######################################################
#																		
# GetWebSiteURL - get the web site URL from the pagelist
#
# Params:	0+ - PageList
#
# Returns:	0 - status
#				1 - error message
#				2 - the URL of the directory of the referring
#					document.  Relative links refer to this
#					directory, and CGI calls should return
#					the customer to this directory.
#				3 - URL of content directory.
#					3 is usually identical to 2, but in the
#					case of SSL, 2 is insecure and 3 is secure.
#					in the future, we could expand 3 to actually
#					refer to a different server.
#
#######################################################

sub GetWebSiteURL
	{
#? ACTINIC::ASSERT($#_ >= 0, "Invalid argument count in GetWebSiteUrl ($#_)", __LINE__, __FILE__);
	
	my (@PageList);
	(@PageList) = @_;
	
	if ($#PageList == -1)
		{
		my ($sDigest,$sBaseFile) = ACTINIC::CaccGetCookies();		# See if the user logged in
		if( !$sBaseFile )
			{
			return ($::FAILURE, "Unable to retrieve web site URL from NULL page list", '', 0);
			}
		else
			{
			$PageList[0] = $sBaseFile;
			}
		}
	
	#######
	# retrieve the web site url
	#######
	my ($nIndex, $sTemp, $sReferenceUrl);
	$sTemp = $PageList[0];								# get the primary reference page
	while ($sTemp =~ /\//g)
		{
		$nIndex = pos $sTemp;							# locate the last "/"
		}
	$sReferenceUrl = substr ($sTemp, 0, $nIndex);	# snag the url
	#
	# If we still don't have correct URL then use the hard coded base URL
	#
	if ($sReferenceUrl eq "/" || $sReferenceUrl eq "")
		{
		$sReferenceUrl = $$::g_pSetupBlob{CATALOG_URL};
		}	
	my $sContentUrl = $sReferenceUrl;
	
	if ($sContentUrl &&									# if the web site url has been defined and
		$$::g_pSetupBlob{USE_SSL})						# we are using SSL security
		{
		$sContentUrl =~ s/http:\/\//https:\/\//i;	# make the images, etc. use secure transfer
		}
		 
	return ($::SUCCESS, '', $sReferenceUrl, $sContentUrl);
	}

#######################################################
#																		
# PrepareRefPageData - prepare the ref page data for
#	insertion into HTML
#
# Params:	0 - original input data
#				1 - pointer to the page list
#				2 - encode flag - If $::TRUE, encode the
#						components of the refpage string
#						before returning it
#
# Returns:	0 - status
#				1 - error message
#				2 - previous query string
#				3 - refpage string
#
#######################################################

sub PrepareRefPageData
	{
#? ASSERT($#_ == 2, "Incorrect parameter count in PrepareRefPageData", __LINE__, __FILE__);

	my ($sPrevQuery, $pPageList, $bEncode) = @_;
	($sPrevQuery) = split (/\&REFPAGE/, $sPrevQuery); # drop any refpage information from the previous query - tracked sep
	#
	# encode the ref page list
	#
	my $sHistoryElement;
	my $sRefPageList;
	if ($bEncode)
		{
		foreach $sHistoryElement (@$pPageList)
			{
			my @Response = EncodeText($sHistoryElement, $::FALSE);
			$sRefPageList .= $Response[1] . '|||';
			}
		}
	else
		{
		$sRefPageList = join('|||', @$pPageList);
		$sRefPageList .= '|||';
		}
	return ($::SUCCESS, '', $sPrevQuery, $sRefPageList);
	}

##############################################################################################################
#																		
# CGI Input Processing (should use CGI.pm but forbidden)- End
#
##############################################################################################################

##############################################################################################################
#																		
# File Read Calls - Begin
#
##############################################################################################################

#######################################################
#																		
# GetSectionBlobName - make the blob name from the ID
#
# Input:    0 - section ID
#
# Returns:	0 - return code ($::SUCCESS or $::FAILURE)
#           1 - error message (if any)
#				2 - blob name
#
#######################################################

sub GetSectionBlobName
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in GetSectionBlobName ($#_)", __LINE__, __FILE__);
	#
	# Validate the input ID - make sure it contains only digits
	#
	if ($_[0] !~ /^(\d+)$/)								# if the section ID does not contain only digits
		{
		return ($::FAILURE, GetPhrase(-1, 306));		# bad input
		}
	my $nID = $1;											# retrieve the ID
	
	return ($::SUCCESS, undef, sprintf('A000%d.cat', $nID));	# format and return the filename
	}

#######################################################
#																		
# GetProduct - locate a product object given its
#	product reference.  if the queried product has
#	been removed from the catalog, GetProduct will
#	return NOTFOUND.
#
# Params:	0 - the product reference
#				1 - the section blob filename
#				2 - file path
#
# Returns:	0 - status (SUCCESS, FAILURE, NOTFOUND)
#				1 - error message
#				2 - a reference to the product
#
#######################################################

sub GetProduct
	{
#? ACTINIC::ASSERT($#_ == 2, "Invalid argument count in GetProduct ($#_)", __LINE__, __FILE__);
	
	my ($ProductRef, $sSectionBlobFilename, $sPath);
	($ProductRef, $sSectionBlobFilename, $sPath) = @_;
	if (length $ProductRef == 0)
		{
		return ($::FAILURE, GetPhrase(-1, 37), 0, 0);
		}
	#
	# see if the section is already in memory
	#
	my ($bInMemory);
	$bInMemory = defined $::g_pSectionList{$sSectionBlobFilename};

	#
	# If the item is not in memory, read the section blob
	#
	my (@Response, $Status, $Message);
	if (!$bInMemory)
		{
		@Response = ReadSectionFile($sPath.$sSectionBlobFilename);
		($Status, $Message) = @Response;
		if ($Status != $::SUCCESS)
			{
			return ($::NOTFOUND, GetPhrase(-1, 173, $ProductRef), \%::g_DeletedProduct);
			}
		my $nVersion = 10;
		if (${$::g_pSectionList{$sSectionBlobFilename}}{VERSION} != $nVersion)		# not the correct blob version number
			{
			return ($::FAILURE, "Section blob version is " . ${$::g_pSectionList{$sSectionBlobFilename}}{VERSION} .
			  ", but only version $nVersion is supported.  File: $sSectionBlobFilename", 0, 0);
			}	
		}
	#
	# see if the product was found in the file.  If not, the supplier must have removed the item from the
	# catalog after we added the item to the cart.
	#

	if (!defined ${$::g_pSectionList{$sSectionBlobFilename}}{$ProductRef})
		{
		return ($::NOTFOUND, GetPhrase(-1, 173, $ProductRef), \%::g_DeletedProduct);
		}
		
	return ($::SUCCESS, '', ${$::g_pSectionList{$sSectionBlobFilename}}{$ProductRef});
	}

#######################################################
#																		
# GetProductReferenceFromVariant - translate the
#	specified product variant code into a product
#	reference.
#
# Params:	0 - the variant code
#				1 - the section blob filename
#				2 - file path
#
# Returns:	0 - status (SUCCESS, FAILURE, NOTFOUND)
#				1 - error message
#				2 - the product reference
#
#######################################################

sub GetProductReferenceFromVariant
	{
#? ACTINIC::ASSERT($#_ == 2, "Invalid argument count in GetProductReferenceFromVariant ($#_)", __LINE__, __FILE__);
	my ($sInvalidProductReference) = "'";
	my ($sVariantCode, $sSectionBlobFilename, $sPath);
	($sVariantCode, $sSectionBlobFilename, $sPath) = @_;
#? ACTINIC::ASSERT(length $sVariantCode > 0, "Invalid product variant code (empty).", __LINE__, __FILE__);
	#
	# see if the section is already in memory
	#
	my ($bInMemory);
	$bInMemory = defined $::g_pVariantList{$sSectionBlobFilename};
	#
	# If the item is not in memory, read the section blob
	#
	my (@Response, $Status, $Message);
	if (!$bInMemory)
		{
		@Response = ReadSectionFile($sPath.$sSectionBlobFilename);
		($Status, $Message) = @Response;
		if ($Status != $::SUCCESS)
			{
			return (@Response);
			}
		my $nVersion = 0;
		if (${$::g_pVariantList{$sSectionBlobFilename}}{VERSION} != $nVersion)	# not the correct blob version number
			{
			return ($::FAILURE, "Variant blob version is " . ${$::g_pVariantList{$sSectionBlobFilename}}{VERSION} .
				", but only version $nVersion is supported.  File: $sSectionBlobFilename", 0, 0);
			}
		}
	#
	# see if the product was found in the file.  If not, the supplier must have removed the item from the
	# catalog after we added the item to the cart.
	#
	if (!defined ${$::g_pVariantList{$sSectionBlobFilename}}{$sVariantCode})
		{
		return ($::FAILURE, GetPhrase(-1, 190, $sVariantCode), $sInvalidProductReference);
		}
		
	return ($::SUCCESS, undef, ${$::g_pVariantList{$sSectionBlobFilename}}{$sVariantCode});
	}

#######################################################
#																		
# ReadSetupFile - read the setup blob file
#
# Params:	0 - path
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	$::g_pSetupBlob - points to the global
#					setup hash
#
#######################################################

sub ReadSetupFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadSetupFile ($#_)", __LINE__, __FILE__);

	#
	# Check if the setup file is already loaded
	#
	if (defined $::g_pSetupBlob &&
		 scalar(keys(%$::g_pSetupBlob)) > 0)
		{
		return ($::SUCCESS, "", 0, 0);					# we are done			
		}
		
	my @Response = ReadConfigurationFile($_[0]."nqset00.fil",'$g_pSetupBlob');	# load the catalog
	if ($Response[0] != $::SUCCESS)
		{
		return (@Response);
		}
		
	my $nSetupVersion = 23;
	if ($$::g_pSetupBlob{VERSION} != $nSetupVersion) # not the correct blob version number
		{
		return ($::FAILURE, "Setup blob version is " . $$::g_pSetupBlob{VERSION} .
			", but only version $nSetupVersion is supported.", 0, 0);
		}
	my $nMinorVersion = 1;
	if ($$::g_pSetupBlob{MINOR_VERSION} < $nMinorVersion) # not the correct blob version number
		{
		return ($::FAILURE, "Setup blob minor version number is " . $$::g_pSetupBlob{MINOR_VERSION} .
			", but minor version $nMinorVersion is required.", 0, 0);
		}
	
	$::g_sRequiredColor	= $$::g_pSetupBlob{REQUIRED_COLOR}; # store the global required field color

	if ($::g_sContentUrl &&								# if the web site url has been defined and
		$$::g_pSetupBlob{USE_SSL})						# we are using SSL security
		{
		$::g_sContentUrl =~ s/http:\/\//https:\/\//i;	# make the images, etc. use secure transfer
		}

	$::g_sAccountScript = $$::g_pSetupBlob{CGI_URL};				# Full HTTP path to account script
	$::g_sAccountScript .= sprintf("bb%6.6d%s",$$::g_pSetupBlob{CGI_ID},$$::g_pSetupBlob{CGI_EXT});
	if( $$::g_pSetupBlob{USE_SSL} )
		{
		$::g_sAccountScript =~ s/http:\/\//https:\/\//i;	# make the script use SSL
		}

	# PRESNET
	# Presnet: set flags by uncommenting these changes
	#
#	$$::g_pSetupBlob{'EMAIL_ORDER'}	= $::TRUE;
#	$$::g_pSetupBlob{'REVERSE_ADDRESS_CHECK'}	= $::TRUE;
#	$$::g_pSetupBlob{'SUPPRESS_CART_WITH_CONFIRM'}	= $::TRUE;
#	$$::g_pSetupBlob{'DISPLAY_CART_AFTER_CONFIRM'}	= $::TRUE;
#	$$::g_pSetupBlob{'PROCEED_CHECKOUT'} = 'pwc.gif';
#	$$::g_pSetupBlob{'CONTINUE_SHOP'} = 'cs.gif';
#	$$::g_pSetupBlob{'EDIT_CART'} = 'ec.gif';
#	$$::g_pSetupBlob{'CONFIRM_IMG'} = 'cnfm.gif';
#	$$::g_pSetupBlob{'CANCEL_IMG'} = 'can.gif';
#	$$::g_pSetupBlob{'REMOVE_IMG'} = 'rem.gif';
#	$$::g_pSetupBlob{'EDIT_IMG'} = 'edit.gif';
	# PRESNET
		 
	return ($::SUCCESS, "", 0, 0);					# we are done
	}

#######################################################
#																		
# ReadCatalogFile - read the catalog blob file.
#
# Params:	0 - path
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	$::g_pCatalogBlob - points to the global
#					catalog hash
#
#######################################################

sub ReadCatalogFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadCatalogFile ($#_)", __LINE__, __FILE__);
	
	my @Response = ReadConfigurationFile($_[0]."A000.cat",'$g_pCatalogBlob');	# load the catalog
	if ($Response[0] != $::SUCCESS)
		{
		return (@Response);
		}
		
	if ($$::g_pCatalogBlob{VERSION} != 2)			# not the correct blob version number
		{
		return ($::FAILURE, "Catalog blob version is " . $$::g_pCatalogBlob{VERSION} .
			", but only version 2 is supported.", 0, 0);
		}
		
	return ($::SUCCESS, "", 0, 0);					# we are done
	}

#######################################################
#																		
# ReadLocationsFile - read the location blob file.
#
# Params:	0 - path
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	$::g_pLocationList - points to the global
#					location hash
#
#######################################################

sub ReadLocationsFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadLocationsFile ($#_)", __LINE__, __FILE__);
	
	my @Response = ReadConfigurationFile($_[0]."locations.fil",'$g_pLocationList');	# load the catalog
	if ($Response[0] != $::SUCCESS)
		{
		return (@Response);
		}
	my $nVersion = 1;
	if ($$::g_pLocationList{VERSION} != $nVersion)	# not the correct blob version number
		{
		return ($::FAILURE, "Location blob version is " . $$::g_pLocationList{VERSION} .
			", but only version $nVersion is supported.", 0, 0);
		}
		
	return ($::SUCCESS, "", 0, 0);					# we are done
	}

#######################################################
#																		
# ReadSearchSetupFile - read the search setup blob file
#
# Params:	0 - path
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	$::g_pSearchSetup - points to the global
#					search setup hash
#
#######################################################

sub ReadSearchSetupFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadSearchSetupFile ($#_)", __LINE__, __FILE__);
	
	my @Response = ReadConfigurationFile($_[0]."search.fil",'$g_pSearchSetup');	# load the catalog
	if ($Response[0] != $::SUCCESS)
		{
		return (@Response);
		}
	my $nVersion = 1;
	if ($$::g_pSearchSetup{VERSION} != $nVersion) # not the correct blob version number
		{
		return ($::FAILURE, "Search setup blob version is " . $$::g_pSearchSetup{VERSION} .
			", but only version $nVersion is supported.", 0, 0);
		}
		
	return ($::SUCCESS, "", 0, 0);					# we are done
	}

#######################################################
#																		
# ReadTaxSetupFile - read the tax blob file.
#
# Params:	0 - path
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	$::g_pLocationList - points to the global
#					location hash
#
#######################################################

sub ReadTaxSetupFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadTaxSetupFile ($#_)", __LINE__, __FILE__);
	
	my @Response = ReadConfigurationFile($_[0]."taxsetup.fil",'$g_pTaxSetupBlob','$g_pTaxZoneMembersTable');	# load the file
	if ($Response[0] != $::SUCCESS)
		{
		return (@Response);
		}
	my $nVersion = 1;
	if ($$::g_pTaxSetupBlob{VERSION} != $nVersion)	# not the correct blob version number
		{
		return ($::FAILURE, "Tax setup blob version is " . $$::g_pTaxSetupBlob{VERSION} .
			", but only version $nVersion is supported.", 0, 0);
		}
		
	return ($::SUCCESS, "", 0, 0);					# we are done
	}

#######################################################
#																		
# ReadSectionFile - read the specified section blob
#	file
#
# Params:	0 - blob filename
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	$::g_pSectionList - points to the global
#					section hash
#
#######################################################

sub ReadSectionFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadSectionFile ($#_)", __LINE__, __FILE__);
	
	my @Response = ReadConfigurationFile(@_,'%g_pSectionList');		# load the configuration
	if ($Response[0] != $::SUCCESS)
		{
		$Response[0] = $::NOTFOUND;					# translate the failure into a product not found error
		return (@Response);
		}
	
	return ($::SUCCESS, "", 0, 0);					# we are done
	}


#######################################################
#																		
# ReadPhaseFile - read phase list
#
# Params:	0 - path
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	$::g_pPhaseList - points to the global
#					phase hash
#
#######################################################

sub ReadPhaseFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadPhaseFile ($#_)", __LINE__, __FILE__);
	
	my @Response = ReadConfigurationFile($_[0]."phase.fil",'$g_pPhaseList');	# load the catalog
	if ($Response[0] != $::SUCCESS)
		{
		return (@Response);
		}
		
	if ($$::g_pPhaseList{VERSION} != 0)				# not the correct blob version number
		{
		return ($::FAILURE, "Phase blob version is " . $$::g_pPhaseList{VERSION} .
			", but only version 0 is supported.", 0, 0);
		}
		
	return ($::SUCCESS, "", 0, 0);					# we are done
	}

#######################################################
#																		
# ReadPromptFile - read the prompt blob
#
# Params:	0 - path
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	$::g_pPromptList - points to the global
#					prompt hash
#
#######################################################

sub ReadPromptFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadPromptFile ($#_)", __LINE__, __FILE__);
	
	my @Response = ReadConfigurationFile($_[0]."prompt.fil",'$g_pPromptList');	# load the catalog
	if ($Response[0] != $::SUCCESS)
		{
		return (@Response);
		}
		
	if ($$::g_pPromptList{VERSION} != 0)				# not the correct blob version number
		{
		return ($::FAILURE, "Prompt blob version is " . $$::g_pPromptList{VERSION} .
			", but only version 0 is supported.", 0, 0);
		}
	#
	# load some common values into globals
	#	
	$::g_sCancelButtonLabel = GetPhrase(-1, 505);
	$::g_sConfirmButtonLabel = GetPhrase(-1, 153);
	$::g_sAddToButtonLabel = GetPhrase(-1, 154);
	$::g_sEditButtonLabel = GetPhrase(-1, 155);
	$::g_sRemoveButtonLabel = GetPhrase(-1, 156);
	$::g_sSearchButtonLabel = GetPhrase(-1, 157);
	#
	# the substitute product for products that have been deleted
	#
	%::g_DeletedProduct =
		(
		'REFERENCE' => ' ',
		'NAME' => ACTINIC::GetPhrase(-1, 174),
		'PRICE' => 0,
		'MIN' => 1,
		'MAX' => 0,
		'TAX_TREATMENT' => $ActinicOrder::ZERO
		);
	#
	# build some index tables to speed generic searches later
	#
	my @keys = keys %{$::g_pPromptList};
	my $list = join(' ', @keys);
	my @scratch = ($list =~ m/([-0-9]+),(\d+) /g);
	while ($#scratch != -1)
		{
		my $nPhraseID = pop @scratch;					# find the next phrase ID
		push (@{$::g_PhraseIndex{pop @scratch}}, $nPhraseID); # add it to the stack for this phase
		}
	return ($::SUCCESS, "", 0, 0);					# we are done
	}

#######################################################
#																		
# ReadConfigurationFile - read the specified blob
#	file
#
# Params:	0 - blob filename
#           1... optional - global variables to be shared with
#                the script
#                Format:  '$foo','$bar',... would share $::foo and $::bar
#                This triggers an attempt to load Safe.pm and eval the
#                script in a Safe compartment. If Safe.pm cannot be loaded
#                eval is used and these arguments are ignored.
#                (See EvalInSafe())
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	the appropriate blob
#
#######################################################

sub ReadConfigurationFile
	{
#? ACTINIC::ASSERT($#_ >= 0, "Invalid argument count in ReadConfigurationFile ($#_)", __LINE__, __FILE__);

	my $sFilename = shift;
	my $pShared   = \@_;					# Optional list of shared variables
	
	my @Response = ReadAndVerifyFile($sFilename);
	if ($Response[0] != $::SUCCESS)
		{
		return(@Response);
		}
	#
	# execute the script (parse the blob)
	#

	if( !$ACTINIC::USESAFE or $#$pShared < 0 )					# No shared variables - use eval
		{
		if (eval($Response[2]) != $::SUCCESS)
			{
			return ($::FAILURE, "Error loading configuration file $sFilename. $@", 0, 0);
			}
		}
	else
		{
		@Response = EvalInSafe($Response[2],$ACTINIC::USESAFEONLY,$pShared);	# Try to use Safe.pm
		if( $Response[0] != $::SUCCESS)
			{
			return ($::FAILURE, "Error loading configuration file $sFilename. $Response[1]", 0, 0);
			}
		}
	
	return ($::SUCCESS, "", 0, 0);					# we are done
	}

#######################################################
#																		
# ReadAndVerifyFile - read the specified script and
#	verify its signature
#
# Params:	0 - filename
#
# Returns:	0 - status
#				1 - error message
#				2 - script 
#
#######################################################

sub ReadAndVerifyFile
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in ReadAndVerifyFile ($#_)", __LINE__, __FILE__);
	
	my ($sFilename);
	($sFilename) = @_;									# set the blob filename
	
	unless (open (SCRIPTFILE, "<$sFilename"))		# open the file
		{
		return ($::FAILURE, "Error opening configuration file $sFilename. $!", 0, 0);
		}
		
	my $nCheckSum = <SCRIPTFILE>;						# read the checksum
	chomp $nCheckSum;										# strip any trailing CRLF
	$nCheckSum =~ s/;$//;								# strip the trailing ;

	my $sScript;
	{
	local $/;
	$sScript = <SCRIPTFILE>;							# read the entire file
	}
	close (SCRIPTFILE);									# close the file
	#
	# calculate the script checksum
	#
	my $uTotal;
		{
		use integer;		
		$uTotal = unpack('%32C*', $sScript);
		}
	#
	# verify the script
	#
	if ($nCheckSum != $uTotal)
		{
		return ($::FAILURE, "$sFilename is corrupt.  The signature is invalid.", 0, 0);
		}

	$sScript =~ s/\r//g;									# remove the dos <CR>

	return ($::SUCCESS, "", $sScript, 0);
	}

################################################################
#
# GetBuyer - retrieve the buyer given the digest
#
# Input:	   0 - digest
#           1 - path
#
# Returns:	0 - status
#           1 - error message if any
#           2 - a reference to the buyer hash
#
################################################################

sub GetBuyer
	{
#? ACTINIC::ASSERT($#_ == 1, 'Incorrect parameter count ACTINIC::GetBuyer(' . join(', ', @_) . ").", __LINE__, __FILE__);
	#
	# Since we typically only retrieve one buyer per execution, it is OK to open the file,
	# do the lookup and close the file.  It is easier to maintain this way.
	#
	my ($sDigest, $sPath) = @_;
	if ($sDigest eq $ACTINIC::BuyerDigest)
		{
		return ($::SUCCESS, undef, \%ACTINIC::Buyer);
		}
	undef %ACTINIC::Buyer;
	undef $ACTINIC::BuyerDigest;
	#
	# Open and prepare the index
	#
	my $rFile = \*BUYERINDEX;
	my $sFilename = $sPath . "oldbuyer.fil";
	my ($status, $sMessage) = InitIndex($sFilename, $rFile, 0);
	if ($status != $::SUCCESS)
		{
		return ($status, $sMessage);
		}
 	eval
		{
		require Digest::MD5;								# Try loading MD5
		import Digest::MD5 'md5_hex';
		};
	if ($@)
		{
		require NETQUOTEVAR:DIGESTPERLMD5;
		import Digest::Perl::MD5 'md5_hex';			# Use Perl version if not found
		}
	#
	# Find the buyer
	#
	my $sUserName = $ACTINIC::B2B->Get('UserName');
	my $sUserHash = md5_hex($sUserName . $sDigest);
	my $sUserKey = $ACTINIC::B2B->Get('UserKey');
	my $sValue;
	($status, $sMessage, $sValue) = IndexSearch($sUserHash, 2, $rFile, $sFilename);
	if ($status != $::SUCCESS)
		{
		CleanupIndex($rFile);
		return ($status, $sMessage);
		}
	CleanupIndex($rFile);
	#
	# Decrypt the index value if $sUserKey is present.
	#
	if ($sUserKey)
		{
		$sUserKey =~ s/([A-Fa-f0-9]{2})/pack("C",hex($1))/ge;
		my @PrivateKey = unpack('C*',$sUserKey);
		my ($sLength, $sDetails) = split(/ /, $sValue);
		$sDetails =~ s/([A-Fa-f0-9]{2})/pack("C",hex($1))/ge;
				
		ActinicEncrypt::InitEncrypt(@{$$::g_pSetupBlob{PUBLIC_KEY_128BIT}});
		$sDetails = ActinicEncrypt::DecryptSafer($sDetails, @PrivateKey);
		$sValue = substr($sDetails,0,$sLength);	# restore it's size to the original length
		}
	#
	# Parse the index value into a hash.  See CCustomerBuyerItem::operator CString for packing details.
	#
   $sValue =~ s/([^ ])$/$1 /;							# if there is no trailing space add one
   $sValue .= 'a';                              # this is used to prevent the split from stripping trailing empty fields
	my @Details = split(/ /, $sValue);
   pop @Details;											# clear the trailing bogus "a"
	my @Labels = qw (ID AccountID Status InvoiceAddressRule InvoiceAddressID DeliveryAddressRule
						  DeliveryAddressID MaximumOrderValue EmailOnOrder LimitOrderValue HideRetailPrices
						  EmailAddress Name Salutation Title TelephoneNumber FaxNumber);
	if( $sUserKey )
		{
		push @Labels,'AccountKey';
		}
#? ACTINIC::ASSERT($#Details == $#Labels, 'Corrupt index ACTINIC::GetBuyer(' . "$#Details != $#Labels).", __LINE__, __FILE__);
	my $nIndex;
	#
	# Load the hash.  Note that Labels and Details are sorted in the same order
	#
	foreach ($nIndex = 0; $nIndex <= $#Details; $nIndex++)
		{
		$ACTINIC::Buyer{$Labels[$nIndex]} = DecodeText($Details[$nIndex], $ACTINIC::FORM_URL_ENCODED);
		}
	if( $sUserKey )
		{
		$ACTINIC::B2B->Set('AccountKey',$ACTINIC::Buyer{AccountKey});
		}

	$ACTINIC::BuyerDigest = $sDigest;				# remember the digest for automated access later

	return ($::SUCCESS, undef, \%ACTINIC::Buyer);
	}

################################################################
#
# GetCustomerAccount - retrieve the customer given the ID
#
# Input:	   0 - ID
#           1 - path
#
# Returns:	0 - status
#           1 - error message if any
#           2 - a reference to the account hash
#
################################################################

sub GetCustomerAccount
	{
#? ACTINIC::ASSERT($#_ == 1, 'Incorrect parameter count ACTINIC::GetCustomerAccount(' . join(', ', @_) . ").", __LINE__, __FILE__);
	#
	# Since we typically only retrieve one account per execution, it is OK to open the file,
	# do the lookup and close the file.  It is easier to maintain this way.
	#
	my ($nID, $sPath) = @_;
	if ($nID == $ACTINIC::AccountID)
		{
		return ($::SUCCESS, undef, \%ACTINIC::Account);
		}
	undef %ACTINIC::Account;
	undef $ACTINIC::AccountID;
	#
	# Open and prepare the index
	#
	my $rFile = \*ACCOUNTINDEX;
	my $sFilename = $sPath . "oldaccount.fil";
	my ($status, $sMessage) = InitIndex($sFilename, $rFile, 0);
	if ($status != $::SUCCESS)
		{
		return ($status, $sMessage);
		}
	#
	# Find the account
	#
	my $sValue;
	($status, $sMessage, $sValue) = IndexSearch($nID, 2, $rFile, $sFilename);
	if ($status != $::SUCCESS)
		{
		CleanupIndex($rFile);
		return ($status, $sMessage);
		}
	CleanupIndex($rFile);
	#
	# If customer accounts are encrypted do the decryption here
	#
	my $sAccountKey = $ACTINIC::B2B->Get('AccountKey');
	if( $sAccountKey )
		{
		$sAccountKey =~ s/([A-Fa-f0-9]{2})/pack("C",hex($1))/ge;
		my @PrivateKey = unpack('C*',$sAccountKey);
		my ($sLength, $sDetails) = split(/ /, $sValue);
		$sDetails =~ s/([A-Fa-f0-9]{2})/pack("C",hex($1))/ge;
		
		ActinicEncrypt::InitEncrypt(@{$$::g_pSetupBlob{PUBLIC_KEY_128BIT}});
		$sDetails = ActinicEncrypt::DecryptSafer($sDetails, @PrivateKey);
		$sValue = substr($sDetails,0,$sLength);	# restore it's size to the original length
		}
	#
	# Parse the index value into a hash.  See CCustomerItem::operator CString and CIndexValueCustomerAccount::operator CString for packing details.
	#
   $sValue =~ s/([^ ])$/$1 /;							# if there is no trailing space add one
   $sValue .= 'a';                              # this is used to prevent the split from stripping trailing empty fields
	my @Details = split(/ /, $sValue);
   pop @Details;											# clear the trailing bogus "a"
	my @Labels = qw (EmailOnOrder InvoiceAddressRule Status InvoiceAddress PriceSchedule DefaultPaymentMethod
						  AccountName EmailAddress TelephoneNumber FaxNumber Name Salutation Title AddressList);
#? ACTINIC::ASSERT($#Details == $#Labels, 'Corrupt index ACTINIC::GetAccount(' . "$#Details != $#Labels).", __LINE__, __FILE__);
	my $nIndex;
	#
	# Load the hash.  Note that Labels and Details are sorted in the same order
	#
	foreach ($nIndex = 0; $nIndex <= $#Details; $nIndex++)
		{
		$ACTINIC::Account{$Labels[$nIndex]} = DecodeText($Details[$nIndex], $ACTINIC::FORM_URL_ENCODED);
		}

	return ($::SUCCESS, undef, \%ACTINIC::Account);
	}

################################################################
#
# GetCustomerAddress - get the customer account address
#
# Input:	   0 - account ID
#           1 - address ID
#           2 - path
#
# Returns:	0 - status
#           1 - error message if any
#           2 - reference address hash
#
################################################################

sub GetCustomerAddress
	{
#? ACTINIC::ASSERT($#_ == 2, 'Incorrect parameter count ACTINIC::GetCustomerAddress(' . join(', ', @_) . ").", __LINE__, __FILE__);
	#
	# Since we occasionally retrieve multiple addresses per execution, we only open the file if it is not open
	# and leave it open until explicitly closed.
	#
	my ($nAccountID, $nAddressID, $sPath) = @_;
	my $sIdentifier = $nAccountID . ":" . $nAddressID;
	if (defined $ACTINIC::Addresses{$sIdentifier})
		{
		return ($::SUCCESS, undef, $ACTINIC::Addresses{$sIdentifier});
		}
	#
	# If the file is not open, open and prepare the index
	#
	my $sFilename = $sPath . "oldaddress.fil";
	if (!defined $ACTINIC::rAddressFileHandle)
		{
		$ACTINIC::rAddressFileHandle = \*ADDRESSINDEX;
		my ($status, $sMessage) = InitIndex($sFilename, $ACTINIC::rAddressFileHandle, 0);
		if ($status != $::SUCCESS)
			{
			return ($status, $sMessage);
			}
		}
	#
	# Find the address
	#
	my ($status, $sMessage, $sValue) = IndexSearch($sIdentifier, 2, $ACTINIC::rAddressFileHandle, $sFilename);
	if ($status != $::SUCCESS)
		{
		CleanupIndex($ACTINIC::rAddressFileHandle);
		undef $ACTINIC::rAddressFileHandle;
		return ($status, $sMessage);
		}
	#
	# If customer addresses are encrypted do the decryption here
	#
	my $sAccountKey = $ACTINIC::B2B->Get('AccountKey');
	if( $sAccountKey )
		{
		$sAccountKey =~ s/([A-Fa-f0-9]{2})/pack("C",hex($1))/ge;
		my @PrivateKey = unpack('C*',$sAccountKey);
		my ($sLength, $sDetails) = split(/ /, $sValue);
		$sDetails =~ s/([A-Fa-f0-9]{2})/pack("C",hex($1))/ge;
		
		ActinicEncrypt::InitEncrypt(@{$$::g_pSetupBlob{PUBLIC_KEY_128BIT}});
		$sDetails = ActinicEncrypt::DecryptSafer($sDetails, @PrivateKey);
		$sValue = substr($sDetails,0,$sLength);	# restore it's size to the original length
		}
	#
	# Parse the index value into a hash.  See CCustomerAddressItem::operator CString for packing details.
	#
   $sValue =~ s/([^ ])$/$1 /;							# if there is no trailing space add one
   $sValue .= 'a';                              # this is used to prevent the split from stripping trailing empty fields
	my @Details = split(/ /, $sValue);
   pop @Details;											# clear the trailing bogus "a"
	my @Labels = qw (ValidAsInvoiceAddress ValidAsDeliveryAddress ExemptTax1 ExemptTax2 CountryCode StateCode Name
						  Line1 Line2 Line3 Line4 PostCode Tax1ExemptData Tax2ExemptData);
#? ACTINIC::ASSERT($#Details == $#Labels, 'Corrupt index ACTINIC::GetCustomerAddress(' . "$#Details != $#Labels).", __LINE__, __FILE__);
	my $nIndex;
	#
	# Load the hash.  Note that Labels and Details are sorted in the same order
	#
	foreach ($nIndex = 0; $nIndex <= $#Details; $nIndex++)
		{
		$ACTINIC::Addresses{$sIdentifier}{$Labels[$nIndex]} = DecodeText($Details[$nIndex], $ACTINIC::FORM_URL_ENCODED);
		}

	return ($::SUCCESS, undef, $ACTINIC::Addresses{$sIdentifier});
	}

################################################################
#
# CloseCustomerAddressIndex - cleanup up the file
#
################################################################

sub CloseCustomerAddressIndex
	{
	if (defined $ACTINIC::rAddressFileHandle)
		{
		CleanupIndex($ACTINIC::rAddressFileHandle);
		undef $ACTINIC::rAddressFileHandle;
		}
	}

################################################################
#
# InitIndex - initialize the specified index file tables
#
# Input:	   0 - the path to the data file
#           1 - a reference to the desired file handle
#           2 - expected file version
#
# Returns:	0 - status
#           1 - error message if any
#
################################################################

sub InitIndex
	{
#? ACTINIC::ASSERT($#_ == 2, 'Incorrect parameter count ACTINIC::InitIndex(' . join(', ', @_) . ").", __LINE__, __FILE__);
	my ($sPath, $rFileHandle, $nExpectedVersion) = @_;
	#
	# Open the index.  Retry a couple of times on failure just incase an update is in progress.
	#
	my ($status, $sError);
	my $nRetryCount = $ACTINIC::MAX_RETRY_COUNT;
	$status = $::SUCCESS;
	while ($nRetryCount--)
		{
		unless (open ($rFileHandle, "<$sPath"))
			{
			$sError = $!;
			sleep $ACTINIC::RETRY_SLEEP_DURATION;	# pause a moment
			$status = $::FAILURE;
			$sError = ACTINIC::GetPhrase(-1, 282, $sPath, $sError);
			next;
			}
		binmode $rFileHandle;
	   #
	   # Check the file version number
	   #
		my $sBuffer;
		unless (read($rFileHandle, $sBuffer, 2) == 2) # read the blob version number (a short)
			{
			$sError = $!;
			close ($rFileHandle);
			return ($::FAILURE, ACTINIC::GetPhrase(-1, 283, $sPath, $sError));
			}

		my ($nVersion) = unpack("n", $sBuffer);	# convert to a number
		if ($nVersion != $nExpectedVersion)
			{
			close($rFileHandle);
			sleep $ACTINIC::RETRY_SLEEP_DURATION;	# pause a moment
			$status = $::FAILURE;
			$sError = ACTINIC::GetPhrase(-1, 284, $sPath, $nExpectedVersion, $nVersion);
			next;
			}

		last;
		}

	return($status, $sError);
	}

################################################################
#
# CleanupIndex - do the cleanup work
#
# Input:	   0 - reference to the index file handle
#
################################################################

sub CleanupIndex
	{
	close ($_[0]);
	}

###############################################################
#
# IndexSearch - search an index for the key.  The result of
#   this recursive function is the index value.  This function
#   assumes that each key has exactly one value.  It can
#   be used for product and account indices.  Search indices
#   where multiple results are possible should use another
#   form of this function.
#
# Input:	   0 - search key (or remaining fragment on
#               recursive call)
#           1 - point to start in the file
#           2 - file handle
#           3 - file path (for identification in errors)
#
# Returns:  0 - status
#           1 - error message
#           2 - value
#
###############################################################

sub IndexSearch
	{
#? ACTINIC::ASSERT($#_ == 3, 'Incorrect parameter count IndexSearch(' . join(', ', @_) . ").", __LINE__, __FILE__);
	my ($sKey, $nLocation, $rFile, $sFileName) = @_;

	my ($nDependencies, $nCount, $nRefs, $sRefs, $sBuff, $sFragment, $sValue);
	my ($nIndex, $sSeek, $nHere, $nLength, $sNext, $nRead);
	# 
   # At the start of the file, we have an (empty) value list
   # followed by a list of dependency records
	# 
	unless (seek($rFile, $nLocation, 0))			# Seek to node
		{
		return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
		}
	# 
   # Read the value (if any).
	# 
	unless (read($rFile, $sBuff, 2) == 2)			# Read the count
		{
		return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
		}

	($nCount) = unpack("n", $sBuff);					# Turn into an integer
	
	for ($nIndex = 0; $nIndex < $nCount; $nIndex++)
		{
		unless (read($rFile, $sBuff, 2) == 2)		# Get value length
			{
			return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
			}

		($nLength) = unpack("n", $sBuff);			# unpack the value length

		unless (read ($rFile, $sValue, $nLength) == $nLength) # read the value
			{
			return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
			}
		
		unless (read($rFile, $sBuff, 1) == 1)		# read the reference count
			{
			return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
			}
		($nRefs) = unpack("C", $sBuff);				# Unpack it

		$sRefs = "";										# Kill left-over references
		if ($nRefs > 0)
			{
			unless (read($rFile, $sRefs, $nRefs) == $nRefs)	# Read and ignore the actual refs
				{
				return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
				}
			}

		if ($sKey eq "")					# If this is an exact match
			{
#? ACTINIC::ASSERT(1 == $nCount, "Index match not unique.", __LINE__, __FILE__);
			return ($::SUCCESS, undef, $sValue);
			}
		}
	# 
   # Now search the dependencies
   #
	unless (read($rFile, $sBuff, 2) == 2)			# Read count
		{
		return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
		}
	$nDependencies = unpack("n", $sBuff);			# Count of dependencies (network short)
	
	for ($nIndex = 0; $nIndex < $nDependencies; $nIndex++)
		{
		unless (read($rFile, $sBuff, 1) == 1)		# Read fragment length
			{
			return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
			}
		$nLength = unpack("C", $sBuff);				# Unpack it

		unless (read($rFile, $sFragment, $nLength) == $nLength) # Read the string fragment
			{
			return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
			}
		unless (read($rFile, $sSeek, 4) == 4)		# Read the link (convert later, if we need it)
			{
			return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
			}
		#
		# We only care about the fragment length as far as
		# the length of the word we're looking for
		#
		$sFragment = substr($sFragment, 0, length($sKey)); # Reduce fragment to useful length
		#
		# Allow special regex characters in $sFragment
		#
		my $sQuotedFragment = quotemeta($sFragment);
		#
		# If the fragment partially matches our word then we
		# continue down the tree. It only needs to match as much
		# of the word as we have - it's perfectly possible for
		# the fragment to be longer than the word
		#
		if ($sKey =~ m/^$sQuotedFragment/) # Does it match?
			{
			$sNext = $';									# Get part after match
			$nHere = tell($rFile);						# Save where we are

			my ($status, $sError, $sValue) = IndexSearch($sNext, unpack("N", $sSeek), $rFile, $sFileName); # Look down tree
			if ($status == $::FAILURE ||				# if the lookup errored or
				 $status == $::SUCCESS)					# if it was completed, 
				{
				return ($status, $sError, $sValue);	# return the state
				}
			#
			# If we are here, $::NOTFOUND was returned, try the next one
			#
			unless (seek($rFile, $nHere, 0))			# Back to where we were
				{
				return ($::FAILURE, ACTINIC::GetPhrase(-1, 285, $sFileName, $!));
				}
			}

		if ($sFragment gt $sKey)						# If we've passed the point in the list
			{
			last;												# Don't look further
			} 
		}

	return ($::NOTFOUND, 'Item not found in index');
	}

#######################################################
#																		
# GetPhrase  - Get the specified phrase and format it.
#
# Params:	0 - phase number
#				1 - prompt number
#				2+ - optional list of arguments supplied
#					to complete string formatting
#
# Returns:	0 - prompt string
#
#######################################################

sub GetPhrase
	{
#? ACTINIC::ASSERT($#_ >= 1, "Invalid argument count in GetPhrase ($#_)", __LINE__, __FILE__);

	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	no strict 'refs';										# this class routine symbolic references
	my ($nPhase, $nPrompt, @args);
	if ($#_ < 1)											# incorrect number of arguments
		{
		$nPhase = -1;										# return parameters not set
		$nPrompt = 12;
		@args = ('GetPhrase');
		}
	else
		{
		($nPhase, $nPrompt, @args) = @_;
		}
	
	my ($sPhrase);
	if (defined $::g_pPromptList)						# if the phrase list is defined and
		{
		$sPhrase = $$::g_pPromptList{"$nPhase,$nPrompt"}{PROMPT};
		}
	elsif (defined $::g_InputHash{"PHRASE$nPhase,$nPrompt"}) # the phrases are in hidden parameters
		{
		$sPhrase = $::g_InputHash{"PHRASE$nPhase,$nPrompt"};
		}
	else
		{
		return ("Phrases not read yet ($nPhase,$nPrompt) {" . join(', ', @args) . "}.");			# report so
		}	
	#
	# process any substitution
	#
	if (defined $sPhrase &&								# if the phrase was found and
		 $#args > -1)										# there are values to substitute
		{
		$sPhrase = sprintf($sPhrase, @args);		# perform the substitution
		}
	
	if (defined $sPhrase)								# if the phrase was defined
		{
		return ($sPhrase); 								# return the phrase
		}

	return ("Phrase not found ($nPhase,$nPrompt) {" . join(', ', @args) . "}!!");
	}

#######################################################
#																		
# GetRequireMessage - retrieve the "this field is required"
#	message for the specified phase and prompt
#
# Params:	0 - phase number
#				1 - prompt number
#
# Returns:	0 - message
#
#######################################################

sub GetRequiredMessage
	{
#? ACTINIC::ASSERT($#_ == 1, "Invalid argument count in GetRequireMessage ($#_)", __LINE__, __FILE__);
	return
			(
			GetPhrase(-1, 55, "\"<B><FONT COLOR=\"" . $::g_sRequiredColor .
			"\">" . GetPhrase($_[0], $_[1]) . "</FONT></B>\"") . "<BR>\n"
			);
	}

#######################################################
#																		
# IsPromptRequired - is the specified prompt required.
#	For simplicity, all errors return $::FALSE.
#
# Params:	0 - phase number
#				1 - prompt number
#
# Returns:	0 - $::TRUE if required
#
#######################################################

sub IsPromptRequired
	{
#? ACTINIC::ASSERT($#_ == 1, "Invalid argument count in IsPromptRequired ($#_)", __LINE__, __FILE__);

	no strict 'refs';										# this class routine symbolic references
	if ($#_ != 1)											# incorrect number of arguments
		{
		return ($::FALSE);
		}
	
	my ($nPhase, $nPrompt) = @_;
	#
	# locate the prompt and return its status
	#
	return ($$::g_pPromptList{"$nPhase,$nPrompt"}{STATUS} == $::REQUIRED ? $::TRUE : $::FALSE); # return it's required status
	}

#######################################################
#																		
# IsPromptHidden - is the specified prompt hidden.
#	For simplicity, all errors return $::FALSE.
#
# Params:	0 - phase number
#				1 - prompt number
#
# Returns:	0 - $::TRUE if hidden
#
#######################################################

sub IsPromptHidden
	{
#? ACTINIC::ASSERT($#_ == 1, "Invalid argument count in IsPromptHidden ($#_)", __LINE__, __FILE__);
	no strict 'refs';										# this class routine symbolic references
	if ($#_ != 1)											# incorrect number of arguments
		{
		return ($::FALSE);
		}
	
	my ($nPhase, $nPrompt) = @_;
	#
	# locate the prompt and return its status
	#
	return ($$::g_pPromptList{"$nPhase,$nPrompt"}{STATUS} == $::HIDDEN ? $::TRUE : $::FALSE); # return it's hidden status
	}

#######################################################
# ChangeAccess                                        
#     Change the access permissions using the various 
#     platform specific calls.                        
#																		
# Params:	0 - the new mode of the file.  supported
#					modes are '' - no permissions,
#					"r" - read only, "rw" - read/write
#         	1 - the file to modify                      
#
# Returns:	number of files changed
#
#######################################################

sub ChangeAccess
	{
# No assert here because ASSERT calls TRACE which calls ChangeAccess - recursion loop
	#
	# !!!!!! This is a function commonly used by many utilities.  Any changes to its interface will
	# !!!!!! need to be verified with the various utility scripts.
	#
	
	my ($mode, $file, $nCount);
	($mode, $file) = @_;

	SecurePath($file);									# make sure only valid filename characters exist in $file to prevent hanky panky
	if ($mode eq '')										# no permissions
		{
		$nCount = chmod 0200, $file;					# process chmod on unix
		}
	elsif ($mode eq "rw")
		{
		$nCount = chmod 0666, $file;					# process chmod on unix
		}		
	elsif ($mode eq "r")
		{
		$nCount = chmod 0644, $file;					# process chmod on unix
		}
	
	return ($nCount);
	}

#######################################################
#
# CleanFileName - Clean iffy characters from file name
#	only letters, digits, '.','_','-' allowed
#	each is changed into '_'
#
#	CAUTION: '/' is not allowed!
#
# Params:	file name
# Returns:	modified name
#
# (rz)
#######################################################

sub CleanFileName
	{
	my $nam = shift;
	$nam =~ tr/a-zA-Z0-9\.\_\-/_/c;
	return $nam;
	}

#######################################################
#
# SecurePath2 - Return an error if the specified path contains
#	any shell characters
#																		
# Input:	   0 - path
#
# Returns:  0 - error or undef
#
#######################################################

sub SecurePath2
	{
	my ($sPath) = $_[0];
	if ($^O eq 'MSWin32')								# NT
		{
		if ($sPath =~ m|[!&<>\|*?()^;\${}\[\]\`\'\"~\n\r]| ||		# the secure path characters (allow backslashes)
			 $sPath =~ m|\0|)
			{
			return("\"$sPath\" contains invalid characters.");
			}
		}
	else
		{
		if ($sPath =~ m|[!&<>\|*?()^;\${}\[\]\`\'\"\\~\n\r]| ||		# the secure path characters (no backslashes)
			 $sPath =~ m|\0|)
			{
			return("\"$sPath\" contains invalid characters.");
			}
		}
	return (undef);
	}

#######################################################
#
# SecurePath - Error out if the specified path contains
#	any shell characters
#																		
# Params:	0 - path
#
#######################################################

sub SecurePath
	{
	my $sError = SecurePath2(@_);
	if ($sError)
		{
		TerminalError($sError);
		}
	}

#######################################################
#
# CheckForShellCharacters - this is not as safe as
#  only tolerating specific characters, but for this
#  release, this is all we have time for.
#																		
# Input:	   0 - value to check
#
# Returns:  0 - error message if any, undef if OK
#
#######################################################

sub CheckForShellCharacters
	{
	my ($sValue) = $_[0];
	if ($sValue =~ m|[!&<>\|*?()^;\${}\[\]\`\'\"\\~\n\r]| ||		# the secure path characters (no backslashes)
		 $sValue =~ m|\0|)
		{
		return ("\"$sValue\" contains invalid characters.");
		}
	return (undef);
	}

#######################################################
#
# GetPath - retrieve the path to the catalog directory
#																		
# Returns:  0 - path
#
#######################################################

sub GetPath
	{
	return ($ACTINIC::s_sPath);
	}

#######################################################
#
# AuthenticateUser - verify the username and password
#  Exits on error.
#																		
# Input:	   0 - user
#				1 - password
#
# Returns:  0 - status
#           1 - message
#
#######################################################

sub AuthenticateUser
	{
	my ($sUsername, $sPassword) = @_;
	my ($sCorrectUsername, $sCorrectPassword) = ('NETQUOTEVAR:USERNAME', 'NETQUOTEVAR:PASSWORD');
	#
	# The username and password must be defined.
	#
	if (!$sUsername ||
		 !$sPassword)
		{
		sleep $ACTINIC::DOS_SLEEP_DURATION;			# Discourage DOS attacks
		return ($::FAILURE, "Undefined Catalog username or password.  Check your Housekeeping | Security settings and try again.  If that fails, try refreshing the site.");
		}
	#
	# Verify the account
	#
	if (!NETQUOTEVAR:ACTINICHOSTMODE)				# stand alone mode
		{
		if ($sUsername ne $sCorrectUsername ||		# either the username or password does not match
			 $sPassword ne $sCorrectPassword)
			{
			sleep $ACTINIC::DOS_SLEEP_DURATION;		# Discourage DOS attacks
			return ($::FAILURE, "Bad Catalog username or password.  Check your Housekeeping | Security settings and try again.  If that fails, try refreshing the site.");
			}
		}
	else														# Actinic Host mode
		{
		#
		# Load the module for access to the configuration files
		#
		eval 'require MallUtil;';
		if ($@)												# the interface module does not exist
			{
			return ($::FAILURE, 'An error occurred loading the MallUtil module.  ' . $@);
			}
		#
		# Retrieve the appropriate record
		#
		my $pShop;
		my ($status, $sError) = MallUtil::GetShopRecordFromUsernameAndPassword($sUsername, $sPassword, \$pShop);
		if ($status == $::BADDATA)
			{
			sleep $ACTINIC::DOS_SLEEP_DURATION;		# Discourage DOS attacks
			return ($status, $sError);
			}
		elsif ($status != $::SUCCESS)
			{
			return ($status, $sError);
			}
		}

	return ($::SUCCESS, undef);
	}

#######################################################
#
# GetLastNonScript - return the last non-script page                                        
#     in a page list.                        
#																		
# Params:	0 - pointer to page list
#
# Returns:	$sRefPage	- last non-script page or the 
#								last page
#
#######################################################

sub GetLastNonScript
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in GetLastNonScript ($#_)", __LINE__, __FILE__);
	my ($sRefPage, $pPageList, $i);
	($pPageList) = @_;
	
	$sRefPage = $$pPageList[-1];						# make sure we return something!!
	#
	# build the pattern for our script name with '\w\w' in place of os, ca, al etc
	#
	my $sScriptURL = sprintf('%s(nph-)?\w\w%6.6d%s', $$::g_pSetupBlob{'CGI_URL'}, $$::g_pSetupBlob{'CGI_ID'},
		$$::g_pSetupBlob{'CGI_EXT'});					# the cart script URL
	#
	# go through the list backwards looking for a page
	# that doesn't match our script reg exp
	#
	for($i = $#$pPageList; $i >= 0; $i--)
		{
		if($$pPageList[$i] !~ m#^$sScriptURL#)		# if the start doesn't look like one of our scripts
			{
			return($$pPageList[$i]);					# assume this is a catalog page
			}
		}
	return($sRefPage);									# return our default page
	}

##############################################################################################################
#																		
# File Read Calls - End
#
##############################################################################################################

##############################################################################################################
#																		
# Blob Write Library - Begin
#
##############################################################################################################

#######################################################
#																		
# OpenWriteBlob - open the blob for write access
#	If the specified filename is empty, use STDOUT.
#	Note that STDOUT mode buffers the message and
#  writes on Close using HTTP header
#
# Params:	0 - filename - if filename == '',
#					then use standard out
#
# Returns:	0 - status
#				1 - error message
#
# Affects:	WBFILE - file handle
#				$s_WBBuffer - file buffer
#				$ACTINIC::s_WBStyle - the blob style
#					= $ACTINIC::FILE - file
#					= $ACTINIC::STDOUT - STDOUT
#					= $ACTINIC::MEMORY - memory
#
#######################################################

sub OpenWriteBlob
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in OpenWriteBlob ($#_)", __LINE__, __FILE__);
	
	my ($sFilename) = @_;
	
	if (length $sFilename > 0 &&						# if we are writting to a file, open it
		 $sFilename ne "memory")
		{
#? 	ACTINIC::ASSERT(undef, "This path is potentially not secure - can we remove it?", __LINE__, __FILE__);
		SecurePath($sFilename);							# make sure only valid filename characters exist in $file to prevent hanky panky
		unless (open (WBFILE, ">$sFilename"))		# open the file
			{
			return ($::FAILURE, "Unable to open $sFilename for writing: $!\n", 0, 0);
			}
		
		binmode WBFILE;									# make sure the file is written in binary mode

		$ACTINIC::s_WBStyle = $ACTINIC::FILE;								# writing to file
		}
	elsif ($sFilename eq "memory")
		{
		$ACTINIC::s_WBBuffer = '';									# clear the buffer
		$ACTINIC::s_WBStyle = $ACTINIC::MEMORY;							# writing to memory
		}
	
	return ($::SUCCESS, '', 0, 0);
	}

#######################################################
#																		
# WriteBlob - write the blob
#
# Params:	0 - \@FieldList - reference to an array
#					of field values to store
#				1 - \@FieldType - ref to an array of field
#					types (in the same order as FieldList
# Returns:	0 - status
#				1 - error message
#
# Expects:	WBFILE - file handle
#
#######################################################

sub WriteBlob
	{
#? ACTINIC::ASSERT($#_ == 1, "Invalid argument count in WriteBlob ($#_)", __LINE__, __FILE__);

	my ($FieldList, $FieldType) = @_;
	
	my ($Field, $Type, @Response, $i);
	for($i = 0; $i <= $#{$FieldList}; $i++)		# loop over the fields in the table
		{
		$Type = $$FieldType[$i];						# the field data type
		$Field = $$FieldList[$i];						# the field value
		
		if ($Type == $::RBBYTE)							# this field is a byte
			{
			@Response = WriteByte($Field);			# Write the byte
			}
		elsif ($Type == $::RBWORD)						# this field is a Word
			{
			@Response = WriteWord($Field);			# Write the Word
			}
		elsif ($Type == $::RBDWORD)					# this field is a double word
			{
			@Response = WriteDoubleWord($Field);	# Write the double word
			}
		elsif ($Type == $::RBQWORD)					# this field is a Java long (64 bits)
			{
			@Response = WriteQuadWord($Field);		# Write the QuadWord
			}
		elsif ($Type == $::RBSTRING)					# this field is a string
			{
			@Response = WriteString($Field);			# Write the string
			}
		else													# unknown field type
			{
			return ($::FAILURE, "Unknown field type $Type\n", 0, 0); # return error
			}
		
		my ($Status, $Message);
		($Status, $Message) = @Response;				# extract the results
		
		if ($Status != $::SUCCESS)						# if the Write failed,
			{
			return ($Status, $Message, 0, 0);		# bail
			}
		}
	
	return ($::SUCCESS, '', 0, 0);
	}

#######################################################
#																		
# CloseWriteBlob - close the blob
#
# Returns:	0 - status
#				1 - error message
#				2 - file buffer
#
# Expects: 	WBFILE - file handle
#				$ACTINIC::s_WBStyle - flag indicating status of WBFILE
#				$ACTINIC::s_WBBuffer - the databuffer (if $ACTINIC::s_WBStyle != $ACTINIC::FILE)
#
#######################################################

sub CloseWriteBlob
	{
	if ($ACTINIC::s_WBStyle == $ACTINIC::FILE)							# file
		{
		close (WBFILE);
		}
	else														# memory
		{
		return ($::SUCCESS, '', $ACTINIC::s_WBBuffer, 0);
		}
	
	return ($::SUCCESS, '', 0);
	}

##############################################################################################################
#																		
# Blob Write Library - End
#
##############################################################################################################

##############################################################################################################
#																		
# Low Level Write Library - Begin
#
##############################################################################################################

#######################################################
#																		
# WriteByte - write a byte
#
# Params:	0 - byte to write
#
# Returns:	0 - status
#				1 - error message
#
# Expects:	$ACTINIC::s_WBStyle - indicating file status
#				WBFILE - if $ACTINIC::s_WBStyle == $ACTINIC::FILE
#				$ACTINIC::s_WBBuffer - if $ACTINIC::s_WBStyle != $ACTINIC::FILE
#
#######################################################

sub WriteByte
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in WriteByte ($#_)", __LINE__, __FILE__);
	
	my ($SIZE, $Byte, $Data);
	$SIZE = 1;												# declare some variables
	($Byte) = @_;
	$Data = 0;
	
	$Data = pack ("C", $Byte);
	
	if ($ACTINIC::s_WBStyle == $ACTINIC::FILE)							# if we are writing to a file
		{
		unless (print WBFILE $Data)					# write the number
			{
			return ($::FAILURE, "Error writing a byte to the file: $!\n", 0);
			}
		}
	else														# if we are dumping to HTTP
		{
		$ACTINIC::s_WBBuffer .= $Data;							# append the data to the buffer
		}
	
	return ($::SUCCESS, '', 0, 0);
	}

#######################################################
#																		
# WriteWord - write a Word in network byte order from
#	the the file
#
# Params:	0 - word to write
#
# Returns:	0 - status
#				1 - error message
#
# Expects:	$ACTINIC::s_WBStyle - indicating file status
#				WBFILE - if $ACTINIC::s_WBStyle == $ACTINIC::FILE
#				$ACTINIC::s_WBBuffer - if $ACTINIC::s_WBStyle != $ACTINIC::FILE
#
#######################################################

sub WriteWord
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in WriteWord ($#_)", __LINE__, __FILE__);
	
	my ($SIZE, $Word, $Data);
	$SIZE = 2;												# declare some variables
	($Word) = @_;
	$Data = 0;
	
	$Data = pack ("n", $Word);
	
	if ($ACTINIC::s_WBStyle == $ACTINIC::FILE)							# if we are dumping to a file
		{
		unless (print WBFILE $Data)					# write the number
			{
			return ($::FAILURE, "Error writing a word to the file: $!\n", 0);
			}
		}
	else														# we are dumping to HTTP
		{
		$ACTINIC::s_WBBuffer .= $Data;							# append to the data buffer
		}
	
	return ($::SUCCESS, '', 0, 0);
	}

#######################################################
#																		
# WriteDoubleWord - write a dword in network byte order
#	from the the file
#
# Params:	0 - double word to write
#
# Returns:	0 - status
#				1 - error message
#
# Expects:	$ACTINIC::s_WBStyle - indicating file status
#				WBFILE - if $ACTINIC::s_WBStyle == $ACTINIC::FILE
#				$ACTINIC::s_WBBuffer - if $ACTINIC::s_WBStyle != $ACTINIC::FILE
#
#######################################################

sub WriteDoubleWord
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in WriteDoubleWord ($#_)", __LINE__, __FILE__);
	
	my ($SIZE, $DWord, $Data);
	$SIZE = 4;												# declare some variables
	($DWord) = @_;
	$Data = 0;
	
	$Data = pack ("N", $DWord);
	
	if ($ACTINIC::s_WBStyle == $ACTINIC::FILE)							# if we are dumping to a file
		{
		unless (print WBFILE $Data)					# write the number
			{
			return ($::FAILURE, "Error writing a double word to the file: $!\n", 0);
			}
		}
	else														# we are dumping to HTTP
		{
		$ACTINIC::s_WBBuffer .= $Data;							# append to the data buffer
		}
	
	return ($::SUCCESS, '', 0, 0);
	}

#######################################################
#																		
# WriteQuadWord - write a Java long (64 bits) in network
#	byte order from the the file
#
# Params:	0 - quad word to write
#
# Returns:	0 - status
#				1 - error message
#
# Expects:	$ACTINIC::s_WBStyle - indicating file status
#				WBFILE - if $ACTINIC::s_WBStyle == $ACTINIC::FILE
#				$ACTINIC::s_WBBuffer - if $ACTINIC::s_WBStyle != $ACTINIC::FILE
#
#######################################################

sub WriteQuadWord
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in WriteQuadWord ($#_)", __LINE__, __FILE__);
	
	my ($SIZE, $QuadWord, $Data);
	$SIZE = 8;												# declare some variables
	($QuadWord) = @_;
	$Data = 0;
	
	my (@Bytes);
	$Bytes[0] = 0;											# 64 bit longs are not really supported
	$Bytes[1] = 0;
	$Bytes[2] = 0;
	$Bytes[3] = 0;
	$Bytes[4] = ($QuadWord & hex("ff000000"))				>> 24;
	$Bytes[5] = ($QuadWord & hex("ff0000"))				>> 16;
	$Bytes[6] = ($QuadWord & hex("ff00"))					>>  8;
	$Bytes[7] = ($QuadWord & hex("ff"));
	
	$Data = pack ("C8", @Bytes);
	
	if ($ACTINIC::s_WBStyle == $ACTINIC::FILE)							# if we are dumping to a file
		{
		unless (print WBFILE $Data)					# write the number
			{
			return ($::FAILURE, "Error writing a 8 byte word to the file: $!\n", 0);
			}
		}
	else														# if we are dumping to HTTP
		{
		$ACTINIC::s_WBBuffer .= $Data;							# append to the data buffer
		}
	
	return ($::SUCCESS, '');
	}

#######################################################
#																		
# WriteString - write a string from the file
#
# Params:	0 - string to write
#
# Returns:	0 - status
#				1 - error message
#
# Expects:	$ACTINIC::s_WBStyle - indicating file status
#				WBFILE - if $ACTINIC::s_WBStyle == $ACTINIC::FILE
#				$ACTINIC::s_WBBuffer - if $ACTINIC::s_WBStyle != $ACTINIC::FILE
#
#######################################################

sub WriteString
	{
#? ACTINIC::ASSERT($#_ == 0, "Invalid argument count in WriteString ($#_)", __LINE__, __FILE__);
	
	my ($String, $Data, $nLength);
	($String) = @_;
	$Data = 0;
	$nLength = length $String;
	
	my (@Response);
	@Response = WriteWord($nLength);					# write the string length
	if (!$Response[0]) 
		{
		return (@Response);								# if there was an error, bail
		}
	
	my ($nByteLength);
	$nByteLength = 2 * $nLength;						# unicode so each char is 2 bytes
	
	if ($nByteLength > 0)								# if there is any string data
		{
		my ($Pack, @Characters);
		
		$Pack = "a".($nByteLength / 2);				# pack the string
		$Data = pack ($Pack, $String);
		
		$Pack = "C".$nByteLength;						# unpack the individual characters
		@Characters = unpack ($Pack, $Data);
		
		$Pack = "xC" x ($nByteLength / 2);			# create the pack string xCxCxC... that writes the unicode string
		$Data = pack ($Pack, @Characters);			# pack the unicode string
		
		if ($ACTINIC::s_WBStyle == $ACTINIC::FILE)			# dumping to file
			{
			unless (print WBFILE $Data)				# write the raw string
				{
				return ($::FAILURE, "Error writing a string to the file: $!\n", 0);
				}
			
			if ($nByteLength > 4096)					# there seems to be a limit on how long of a string we can write
				{
				return ($::FAILURE, "Error writing a string from the file: string is ".
						  "\n\tlonger than 4K - probably bad format or bad version\n", 0);
				}
			}
		else													# dumping to HTTP
			{
			$ACTINIC::s_WBBuffer .= $Data;						# append to the buffer
			}
		}
	
	return ($::SUCCESS, '');
	}

##############################################################################################################
#																		
# Low Level Write Library - End
#
##############################################################################################################

############################################################
#  EvalInSafe - Eval script Safely
#  This function attempts to load Safe.pm module.
#  If succesful it will execute supplied script in a Safe
#  container (no system calls are allowed and only specified
#  variables are shared).
#  Otherwise 'force' switch (second argument) is checked.
#  If true - nothing is done and $::FAILURE is returned
#  If false - eval is used to evaluate script.
#  
#  Arguments
#    0 - script to eval
#    1 - force switch
#    2 - reference to a list of variables to be shared with
#           the script.
#    (format: ('$foo','$bla') will allow sharing $::foo and $::bla)
#    ($::SUCCESS and $::FAILURE are shared automatically)
#  Returns
#    0 - status
#    1 - error message string
#  
#  If eval is executed (either in Safe or using eval) then
#  status and error message are passed from eval.
#  (If error message exists then status is always $::FAILURE)
#  If 'force' flag is on and Safe.pm is not found then status
#  if $::FAILURE.
#  If Safe.pm detects a violation status is $::FAILURE
#
#  Example:
#  If $scr contains following script:
#  $::gtext = "This is my text";
#  $::gtext1 = "This is my text1";
#  return $::SUCCESS;
#  
#  then
#  my ($Result, $Msg ) = EvalInSafe($scr,$::FALSE,'$gtext');
#  
#  will return $::SUCCESS
#    If Safe.pm exists then $::gtext is set but $::gtext1 is not set
#    If Safe.pm does not exist then both are set
#  
#  my ($Result, $Msg ) = EvalInSafe($scr,$::FALSE,'$gtext','$gtext1');
#  
#  returns $::SUCCESS and sets $::gtext and $::gtext1
#  
#  my ($Result, $Msg ) = EvalInSafe($scr,$::TRUE,'$gtext');
#  
#  returns $::SUCCESS and sets $::gtext if Safe.pm exists
#  returns $::FAILURE and sets nothing otherwise
#
#  Ryszard Zybert  Mar 24 18:09:02 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub EvalInSafe
	{
	return ::EvalInSafe(@_);
	}

#------------------------------------------- Start of main ---------

package main;

############################################################
#  EvalInSafe
#    See comments in ACTINIC::EvalInSafe
#  This is here only in order to make it global in order to work
#  with Safe.pm version 1 (using share() instead of share_from())
#
#  Ryszard Zybert  Aug 12 10:48:31 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub EvalInSafe
	{
	my $sScript = shift;			# Script to eval
	my $bForce  = shift;			# If true do it only with Safe
	my $pShare  = shift;			# Reference to a list of shared variables
	my $Result;

	eval 'require Safe';			# Try to load Safe.pm
	if( $@ )							# Cannot find it
		{
		if( $bForce )				# If unsafe eval is forbidden
			{
			return ($::FAILURE, "Cannot load Safe.pm");	# return failure
			}
		$Result = eval($sScript);	# otherwise just use eval
		}
	else								# Safe found
		{
		my $pCnt = new Safe();											# Safe container
		$pCnt->share('$SUCCESS','$FAILURE');						# Always share these two
		$pCnt->share(@$pShare);											# Share variables specified
		$Result = $pCnt->reval($sScript);							# Eval the script
		}
	if( $@ )							# If there is an error message it is a failure
		{
		$Result = $::FAILURE;
		}
	return ($Result,$@);			# done
	}

package ACTINIC;

#------------------------------------------- End of main ---------

#######################################################
#																		
# TRACE - debug trace function.  Works either with
#	server on port 9876 or a file
#
# Params:	sprintf ready list containing message.
#					a newline is automatically appended
#
#######################################################

#?sub TRACE
#?	{
#? $| = 1;
#?	my $sMessage = sprintf(shift, @_);
	
#?	while ($ACTINIC::s_bTraceSockFirstPass)
#?		{
#?		$ACTINIC::s_bTraceSockFirstPass = $::FALSE;
		
#?		my ($remote,$port, $iaddr, $paddr, $proto, $line);
		
#?		$remote  = 'localhost';
#?		$port    = 9876;
#?		if ($port =~ /\D/)
#?			{
#?			$port = getservbyname($port, 'tcp')
#?			}
#?		if (!$port)
#?			{
#?			last;
#?			}
#?		if (!($iaddr = inet_aton($remote)))
#?			{
#?			last;
#?			}
#?		$paddr   = sockaddr_in($port, $iaddr);
		
#?		$proto   = getprotobyname('tcp');
#?		no strict 'subs';
#?		if (!socket(DBOUT, PF_INET, SOCK_STREAM, $proto))
#?			{
#?			last;
#?			}
#?		if (!connect(DBOUT, $paddr))
#?			{
#?			last;
#?			}
		
#?		$ACTINIC::s_bTraceSocket = $::TRUE;

#?		print DBOUT "\n\n";
#?		}

#?	while (!$ACTINIC::s_bTraceSocket &&
#?			  $ACTINIC::s_bTraceFileFirstPass)
#?		{
#?		$ACTINIC::s_bTraceFileFirstPass = $::FALSE;
#?		my $sFilename = GetPath() . 'output.txt';
#?		ChangeAccess('rw', $sFilename);
#?		SecurePath($sFilename);								# make sure only valid filename characters exist in $file to prevent hanky panky
#?		open (DBOUT, ">$sFilename");
#?		}
	
#?	print DBOUT $sMessage . "\n";
#?	}

#######################################################
#																		
# ASSERT - debug ASSERT function.
#
# Params:	0 - condition - if false, throw an assertion
#				1 - message associated with the assertion
#				2 - line number where assertion is
#				3 - file containing assertion
#
#######################################################

#?  sub ASSERT
#?  	{
#?  	my ($bTest, $sMessage, $nLine, $sFile) = @_;
#?  	if (!$bTest)
#?  		{
#?  		$ACTINIC::AssertIsActive = $::TRUE;
#?  		my $sText = 'Assertion failed: ' . $sMessage . ' (' . $sFile . ', line: ' . $nLine . ')';

#?  		if ($ACTINIC::AssertIsLooping)
#?  			{
#?  			my $sCallStack;
#?  			$sCallStack = CallStack();
#?  			TRACE($sText . $sCallStack);
#?  			exit;
#?  			}
#?  		$ACTINIC::AssertIsLooping = $::TRUE;

#?  		TRACE($sText);
#?  		my $sCallStack;
#?  		$sCallStack = CallStackHTML();
#?  		TerminalError($sText . $sCallStack);
#?  		}
#?  	}

#######################################################
#																		
# CallStack - produce a call stack
#
# Returns: string call stack formatted as follows:
#
#	main, line number
#  function, line number
#  next function, line number
#	...
#	current function, line number
#
#######################################################

#?sub CallStack
#?	{
#?	my @call = caller(1);
#?	my $line = $call[2];
#?	my $cnt = 2;

#?	my @stack;

#?	while (defined($call[0]))
#?		{
#?		my $caller = $call[0];
#?		@call = caller($cnt);
#?		$call[3] = $caller if (!defined($call[3]));
#?		unshift(@stack, $call[3] . ", " . $line);
#?		$line = $call[2];
#?		$cnt++;
#?		}
#?	return(join("\r\n", @stack));
#?	}

#######################################################
#																		
# CallStackHTML - produce a call stack in HTML ready format
#
# Returns: string call stack formatted in HTML as follows:
#
#	Call Stack:
#
#		*main*, line number
#  	*function*, line number
#  	*next function*, line number
#		...
#		*current function*, line number
#
#	The actual stack is indented with block quote and the
#	function names are emboldened.
#
#######################################################

#?sub CallStackHTML
#?	{
#?	my $sCallStack = "<BR><BR>Call Stack:<BLOCKQUOTE><B>" . CallStack() . "</BLOCKQUOTE>";
#?	$sCallStack =~ s/\r\n/<BR>\r\n<B>/g;
#?	$sCallStack =~ s/,/\<\/B\>,/g;
#?	return($sCallStack);
#?	}

#######################################################
#
# Search highlighting function
#
#######################################################

###############################################################
#
# HighlightWords - highlight the specified words in the HTML
#   page using the supplied markup.
#
# Input:	   0 - space separated list of words to highlight
#           1 - highlight start markup
#           2 - highlight end markup
#In/Output: 3 - reference to the HTML to be modified.  The
#               modification is done in place.
#
####### WARNING WARNING WARNING WARNING WARNING ###############
#
# The following code does not follow normal Actinic coding standards.
# This is code for Perl experts at 11AM after a strong pot of coffee!
# It is a special case self-modifying code with run-time generation.
#
# It is strongly recommended that you review pages 72-73 of the Blue
# Camel. This uses "s'PATTERN'CODE_TO_CREATE_REPLACEMENT'gesi" and
# exploits the special properties of a "single quote" as a delimiter.
#
####### WARNING WARNING WARNING WARNING WARNING ###############
#
###############################################################

sub HighlightWords
   {
#? ACTINIC::ASSERT($#_ == 3, "Incorrect parameter count HighlightWords(" . join(', ', @_) . ").", __LINE__, __FILE__);
   my ($sWords, $sStart, $sEnd, $rsHTML) = @_;
   #
   # Now, Highlighting words...
   #
   my @Patterns = ();
   #
   # All strings should be preprocessed to single-space delimiters
   # But split on whitespace in case the processing is imperfect
   #
   my @Words = split /\s+?/,$sWords;
   for (@Words)
      {
      # Match at beginning of words. This should handle single character
      # words without problems - but it won't work for all hyphenated words
      # or those containing an apostrophe because those are not in "\b".
      #
      # But the real problem is that $sWords is not entity-escaped like the
		# HTML and hence: "O'Reilly" ne "O&#39;Reilly". Note those are XML/SGML
      # entities, not older HTML "%xx" ones. Ben is already relying on "$_".
      #
      s/\'/\&#39;/g;				# apostrophe in match pattern: O'Reilly
      s/-/\&#45;/g;				# hyphen in match pattern: Diffie-Hellman
      s/\./\&#46;/g;				# period in match pattern: www.actinic.com
      s/_/ /g;						# convert '_' to space in pattern: Big_A_Auto
      #
      # Conversions we don't want to consider: !&;:$%*
      #
		# If an integer, avoid highlight breaking an XML character entity
		# like "&#123;". This is not yet perfect - a ";" following the
		# integer prevents highlighting it even if not XML. The best fix
		# requires unrolling the global replace and explicitly checking
		# if each match is part of an entity. But this should cover most
		# of the practical cases.
		#
		# Highlight extends to word boundaries in all cases.
		#
		if ($_ =~ m/^\d+$/)
			{
			# Make sure word boundary is not ";" after Integer
			#
      	push @Patterns, "\\b$_\[^;\]*?\\b(?!;)";
			}
		elsif ($_ ne '')
			{
      	push @Patterns, "\\b$_.*?\\b";
			}
      }
   #
   # Original George comment retained intact - Bill Birthisel, 11 May 2000
	#
   # do clever stuff (this is a literal quote from Ben's code - if you understand it, comment here)
   #
   # You asked for it! -bill
   #
   # we don't want to alter the title of the page, so we save a copy
   #
   $$rsHTML =~ m~(<title>.+?</title>)~is;
   my $sOldTitle = $1;
   #
   # We substitute an "unlikely magic token" for the title. Since the '!' is
	# a delimiter in advanced search, the token should stay as unlikely as it
	# was before - note: this makes "PROTECTTITLE" a reserved word.
   #
   $$rsHTML =~ s~(<title>.+?</title>)~!PROTECTTITLE!~gis;
   my $sPattern;
   #
   # The patterns from above, i.e. words beginning with the match string.
   # The regex "\b$_.*?\b" is really more like "\b\w+?" in the case of
   # words for which we have not explicitly fixed the escaped characters.
   #
   foreach $sPattern (@Patterns)
      {
      $$rsHTML =~ s'\>(.*?)\<'
         #
         # see WARNING above...this is the start of code generation
         # the indentation is correct - single quote is a delimiter like "{"
         #
         # Extract the text between adjacent markup tags into $1.
         # Many end up ($1 eq "") or ($1 =~ /\s/), but Perl should
         # quickly search-mismatch on those anyway via length checking.
         # Since this is a global replace, it will essentially reread
         # the HTML for each pattern - ok for $#Patterns of 1 or 2, if
         # we allow many more than that, the loop should be reconsidered.
         # For highlighting, more than 3 or 4 may be unworkable anyway.
         # But an advanced search with one match out of a much larger
         # group of patterns could occur.
         #
         my $t = $1;
         $t =~ s/($sPattern)/$sStart$1$sEnd/gsi;
         #
         # Re-insert the text, now surrounded by highlight on/off, between
			# the original markup tags with markup delimiters since the original
			# used them to search
         #
         ">$t<";
         #
         # see WARNING above...next line is the finish
      'gesi;                                    # ' # This single quote eliminates formatting problems with emacs
      }
   # Restore the original title. I can't imagine why he made it a global
   # replace - the whole idea relies on uniqueness. But matches above.
   #
   $$rsHTML =~ s~!PROTECTTITLE!~$sOldTitle~gis;
   }

###############################################################
#
# DeterminePricesToShow - Work out which prices to show
#
# Returns:	($bShowRetailPrices, $bShowCustomerPrices, $nAccountSchedule)
#				$::TRUE or $::FALSE for bShowXXX
#				if $bShowCustomerPrices, then $nAccountSchedule contains schedule ID
#
###############################################################

sub DeterminePricesToShow
	{
	#
	# Need to work out which prices to show
	#
	my $nAccountSchedule = -1;
	my $bShowCustomerPrices = $::FALSE;
	my $bShowRetailPrices = $::TRUE;
	#
	# See if this is a customer account
	#
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');
	if($sDigest ne '')
		{
		#
		# Get the buyer
		#
		my ($Status, $Message, $pBuyer) = ACTINIC::GetBuyer($sDigest, ACTINIC::GetPath()); # look up the buyer
		if ($Status == $::SUCCESS)
			{
			#
			# Got the buyer so get the account
			#
			my $pAccount;
			($Status, $Message, $pAccount) = ACTINIC::GetCustomerAccount($$pBuyer{AccountID}, ACTINIC::GetPath());
			if ($Status == $::SUCCESS)
				{
				#
				# Got the account so get the schedule and whether retail prices are shown
				#
				if( $pAccount->{PriceSchedule} != $ActinicOrder::RETAILID ) # If retail  is not default
					{
					$nAccountSchedule = $pAccount->{PriceSchedule};		# save the schedule
					$bShowRetailPrices = !$pBuyer->{HideRetailPrices};	# save whether we show retail prices
					$bShowCustomerPrices = $::TRUE;							# we show customer prices
					}
				}
			}
		}
	return($bShowRetailPrices, $bShowCustomerPrices, $nAccountSchedule);
	}

###############################################################
#
# GetVariantList - Get the variant list and product ref HTML for component
#
# Input:		[0] - the product reference
#
# Returns:	($VariantList, $sLine)
#				$VariantList - reference to a list of variants
#				$sLine	- HTML for product ref
#
###############################################################

sub GetVariantList
	{
	my ($sProductRef) = @_;
	my ($VariantList, $sLine, $k, $i);
	foreach $k (keys %::g_InputHash)
		{
		if( $k =~ /^(_?)$sProductRef\_/ )
			{
			my $sVariantSpec = $';
			my $cnt = $sVariantSpec =~ tr/_/_/; 			# Count underscores
			if( $cnt == 0 )										# Nothing - we take VALUE
				{
				$VariantList->[$sVariantSpec] = $::g_InputHash{$k};
				$sLine .= "<INPUT TYPE=HIDDEN NAME=\"$sProductRef" . "_" . "$sVariantSpec\" VALUE=\"$::g_InputHash{$k}\">";
				}
			elsif( $cnt == 1 )									# Just one - a simple case
				{
				my ($sAttribute,$sValue) = split('_',$sVariantSpec);
				$VariantList->[$sAttribute] = $sValue;
				$sLine .= "<INPUT TYPE=HIDDEN NAME=\"$sProductRef" . "_" . "$sAttribute\" VALUE=\"$sValue\">";
				}
			else														# More than one - several attributes in one widget
				{
				my @sVarSpecItems = split('_',$sVariantSpec);
				for( $i=0; $i<=$#sVarSpecItems; $i+=2)
					{ 
					$VariantList->[$sVarSpecItems[$i]] = $sVarSpecItems[$i+1];
					$sLine .= "<INPUT TYPE=HIDDEN NAME=\"$sProductRef" . "_" . "$sVarSpecItems[$i]\" VALUE=\"$sVarSpecItems[$i+1]\">";
					}
				}
			}
		}
	return($VariantList, $sLine);
	}

#######################################################
#
# Customer accounts common functions
#
#######################################################

#######################################################
#
# CaccGetCookies - get busines  cookies
#
# Returns  0 - account cookie value
#          1 - base file cookie value
#
# Other fields are stored in the B2B object with B2B->Set()
#######################################################

sub CaccGetCookies
	{
	my ($sCookie, $sCookies);

	my $sReferer = ACTINIC::GetReferrer();
	$sReferer =~ s/\?.*//;							# We need here just the file name information
	
	if(($::g_InputHash{USER} and $::g_InputHash{HASH} and !$::g_InputHash{ORDERHASH}))	# LOGIN page - emulate cookies		
		{
		return ($ACTINIC::B2B->Get('UserIDCookie'),$ACTINIC::B2B->Get('BaseFile'));
		}
	if( $::g_InputHash{OBIREQID} )				# ConnectTrade login page - emulate cookies		
		{
		return ($ACTINIC::B2B->Get('UserIDCookie'),$ACTINIC::B2B->Get('BaseFile'));
		}		
	if( ACTINIC::IsStaticPage($sReferer) )		# If this came from another static page - this is not B2B
		{
		$ACTINIC::B2B->Clear('BrowseID');
		$ACTINIC::B2B->Clear('PostBackURL');
		$ACTINIC::B2B->Clear('BaseFile');
		$ACTINIC::B2B->Clear('UserIDCookie');
		$ACTINIC::B2B->Clear('UserName');
		$ACTINIC::B2B->Set('ClearIDCookie','CLEAR');	# Clear User Cookie next time
		$ACTINIC::B2B->Set('ClearUserCookie','CLEAR');	# Clear User Name Cookie next time
		return ('','');
		}
	$sCookies = $::ENV{'HTTP_COOKIE'};				# try to retrieve the cookie
	my (@CookieList) = split(/;/, $sCookies);		# separate the various cookie variables in the list
	my ($sDigest,$sBaseFile);

	foreach $sCookie (@CookieList)
		{
		$sCookie =~ s/^\s*//;										# strip leading white space
		if( $sCookie =~ /^ACTINIC_BUSINESS/ )					# found the account cookie
			{
			my ($sLabel, $sCookieValue) = split (/=/, $sCookie);		# retrieve the value
			#
			# strip any trailing or leading quotes and spaces
			#
			$sCookieValue =~ s/^\s*\"?//;
			$sCookieValue =~ s/\"?\s*$//;
			my $sCookieText = ACTINIC::DecodeText($sCookieValue, $ACTINIC::FORM_URL_ENCODED);
			#
			# There is one field per line: name TAB value
			#
			my (@Fields) = split("\n",$sCookieText);			# Get all fields
			my $sField;
			foreach $sField (@Fields)								# Extract name and value from each field
				{
				my ($sName,$sData) = split("\t",$sField);		# Split into name and value
				#
				# strip any trailing or leading quotes and spaces
				#
				$sData =~ s/^\s*\"?//;
				$sData =~ s/\"?\s*$//;
				if( $sData eq "" )
					{
					next;
					}
				for ($sName)
					{
					/^ACCOUNT/ and do									# found the account cookie
						{
						$sDigest = $sData;
						last;
						};
					/^BASEFILE/ and do								# found the base file
						{
						$sBaseFile = $sData;
						last;
						};
					/^USERNAME/ and do								# found user name
						{
						$ACTINIC::B2B->Set('UserName',$sData);
						last;
						};
					/^PRODUCTPAGE/ and do							# found the last page shown
						{
						$ACTINIC::B2B->Set('ProductPage',$sData);
						last;
						};
					/^CHALLENGE/ and do								# found the challenge
						{
						$ACTINIC::B2B->Set('UserKey',$sData);
						last;
						};
					/^OBIPOSTBACKURL/ and do							# found the PostBack URL
						{
						$ACTINIC::B2B->Set('PostBackURL',$sData);
						last;
						};
					/^OBIREQID/ and do								# found the BrowseID
						{
						$ACTINIC::B2B->Set('BrowseID',$sData);
						last;
						};		
					/^CID/ and do										# found the CID
						{
						$ACTINIC::B2B->Set('CID',$sData);
						last;
						};		
					/^CNAME/ and do									# found the CNAME
						{
						$ACTINIC::B2B->Set('CNAME',$sData);
						last;
						};								
					last;
					}
				}
			last;															# Found what we were looking for
			}
		}
	if( !$sDigest )											 		# If there is no Digest clear all other user variables
		{
		$ACTINIC::B2B->Clear('BaseFile');
		$ACTINIC::B2B->Clear('UserIDCookie');
		$ACTINIC::B2B->Clear('UserName');
		$ACTINIC::B2B->Clear('UserDigest');
		$ACTINIC::B2B->Clear('ProductPage');
		$ACTINIC::B2B->Clear('PostBackURL');
		$ACTINIC::B2B->Clear('BrowseID');
		$ACTINIC::B2B->Clear('CID');
		$ACTINIC::B2B->Clear('CNAME');
		return ('','');
		}
	return ($sDigest,$sBaseFile);
	}

############################################################
#  CAccBusinessCookie - create Business cookie
#  Uses data stored in B2B object to construct business cookie.
#  Returns encoded cookie value ready for transmission.
#
#  Ryszard Zybert  Aug 31 12:39:42 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub CAccBusinessCookie
	{
	if ( $ACTINIC::B2B->Get('ClearIDCookie') )							# Request to clear
		{
		return ("");
		}
	my $sCookie = "";
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');						# User Digest
	if ( $sDigest )
		{
		if ( $sDigest eq "." )													# '.' is like '' but is TRUE
			{
			$sDigest = "";
			}
		if( $sDigest eq "" )
			{
			return ("");
			}
		$sCookie .= "ACCOUNT\t$sDigest\n";									# Store Digest
		}
	else
		{
		return ("-");	  															# No user - don't bother with cookies
		}
	if ( $ACTINIC::B2B->Get('ClearUserCookie') )
		{
		$sCookie .= "USERNAME\t\n";									 		# Clear username  cookie
		}
	else
		{
		$sCookie .= "USERNAME\t" . $ACTINIC::B2B->Get('UserName') . "\n";					# User name
		}		
	$sCookie .= "BASEFILE\t" . 	$ACTINIC::B2B->Get('BaseFile') .	"\n";					# Basefile
	$sCookie .= "PRODUCTPAGE\t" . $ACTINIC::B2B->Get('ProductPage') .	"\n";				# Productpage
	$sCookie .= "CHALLENGE\t" . 	$ACTINIC::B2B->Get('UserKey') . "\n";					# Challenge
	#
	# ConnectTrade related cookie content
	#
	$sCookie .= "OBIPOSTBACKURL\t" . $ACTINIC::B2B->Get('PostBackURL') . "\n";				# PostBack URL
	$sCookie .= "OBIREQID\t" . 	$ACTINIC::B2B->Get('BrowseID') . "\n";					# BrowseID
	$sCookie .= "CID\t" . 			$ACTINIC::B2B->Get('CID') . "\n";						# Supplier identifier
	$sCookie .= "CNAME\t" . 		$ACTINIC::B2B->Get('CNAME') . "\n";						# name
	return (ACTINIC::EncodeText2($sCookie,0));
	}

#######################################################
# CAccLogin - User login
# No arguments
# Requires: $::g_InputHash needs to be set before
#           $::ENV{HTTP_REFERER}
#           Phrases blob read in
# 
# In this case $::g_sB2BUserIDCookie and $::g_sBaseFile are set.
# Otherwise check ACCOUNT cookie against user list.
# Returns only on success.
#######################################################

sub CAccLogin
	{
	my ($sDigest,$sBaseFile,$Md5);

	$ACTINIC::B2B->Clear('UserIDCookie');
	#
	# When BrowseID is defined then it must be a connecttrade connection
	# So lets save the BrowseID into the business  cookie
	#
	if( $::g_InputHash{OBIREQID} )	
		{
		#
		# Check mandatory fields
		#
		if (!$::g_InputHash{OBIPOSTBACKURL})
			{
			ACTINIC::ReportError("OBIPOSTBACKURL must be defined for ConnectTrade connections!", ACTINIC::GetPath());
			}
		if (!$::g_InputHash{CID})
			{
			ACTINIC::ReportError("VAT number must be defined for ConnectTrade connections!", ACTINIC::GetPath());
			}
		if (!$::g_InputHash{CNAME})
			{
			ACTINIC::ReportError("C. name must be defined for ConnectTrade connections!", ACTINIC::GetPath());
			}
		#
		# If all valid then store it in B2B which will be used for cookie creation
		# Note thet this is the only one phase where this parameters can be found 
		# in the input hash. So all values required by the OBI creation should be
		# grabbed here and stored in the business cookie (the changes should be
		# reflected in CAccBusinessCookie and CaccGetCookies).
		#
		$ACTINIC::B2B->Set('BrowseID', $::g_InputHash{OBIREQID});
		$ACTINIC::B2B->Set('PostBackURL', $::g_InputHash{OBIPOSTBACKURL});
		$ACTINIC::B2B->Set('BaseFile', $$::g_pSetupBlob{CATALOG_URL});	# set basefile
		$ACTINIC::B2B->Set('CID', 		$::g_InputHash{CID});		
		$ACTINIC::B2B->Set('CNAME', 	$::g_InputHash{CNAME});	
		}
	elsif ($::g_InputHash{USER}) 
		{
		#
		# When the BrowseID is not defined but the USER is defined
		# (query comes from the login page)
		# Then clear connecttrade related cookie content
		#
		$ACTINIC::B2B->Clear('PostBackURL');
		$ACTINIC::B2B->Clear('BrowseID');
		}
	#
	# ConnectTrade connection supplies the plain text password
	# for convience (SSL sites)
	# So lets allow plain text passwords
	#
	if( $::g_InputHash{USER} and $::g_InputHash{PASS} and $::g_InputHash{PASS} ne '' )	
		{
		eval
			{
			require Digest::MD5;								# Try loading MD5
			import Digest::MD5 'md5_hex';
			};
		if ($@)
			{
			require NETQUOTEVAR:DIGESTPERLMD5;
			import Digest::Perl::MD5 'md5_hex';			# Use Perl version if not found
			}
		my $sPass = md5_hex($::g_InputHash{PASS});
		$sDigest = md5_hex($::g_InputHash{USER} . $sPass);
		$::g_InputHash{HASH} = $sDigest;
		$ACTINIC::B2B->Set('UserIDCookie',$sDigest);
		$ACTINIC::B2B->Set('UserDigest',$sDigest);
		$ACTINIC::B2B->Set('UserName',$::g_InputHash{USER});
		my $sChallenge;
		if( $::g_InputHash{challenge} )
			{
			$sChallenge = md5_hex($::g_InputHash{challenge} . $sPass);
			}
		else
			{
			$sChallenge = md5_hex($$::g_pSetupBlob{CHALLENGE} . $sPass);
			}		
		$ACTINIC::B2B->Set('UserKey', $sChallenge);
		}
	#
	# If there is not password field defined then procees on the normal way
	#
	elsif( $::g_InputHash{USER} and $::g_InputHash{HASH} )	# If this came from LOGIN page
		{
		$sDigest = $::g_InputHash{HASH};
		$ACTINIC::B2B->Set('UserIDCookie',$sDigest);
		$ACTINIC::B2B->Set('UserName',$::g_InputHash{USER});
		$ACTINIC::B2B->Set('BaseFile',ACTINIC::GetReferrer());
		if( $::g_InputHash{challengeout} )						# Try new and old version for compatibility
			{
			$ACTINIC::B2B->Set('UserKey',$::g_InputHash{challengeout});
			}
		elsif( $::g_InputHash{challenge} )
			{
			$ACTINIC::B2B->Set('UserKey',$::g_InputHash{challenge});
			}
		else
			{
			$ACTINIC::B2B->Set('UserKey',$$::g_pSetupBlob{CHALLENGE});
			}
		}
	else
		{
		my $sReferer = ACTINIC::GetReferrer();
		$sReferer =~ s/\?.*//;							# We need here just the file name information

		if( ACTINIC::IsStaticPage($sReferer) )		# If this came from another static page - this is not B2B
			{
			$sDigest = "";									# Clear everything
			$ACTINIC::B2B->Clear('BaseFile');
			$ACTINIC::B2B->Clear('UserIDCookie');
			$ACTINIC::B2B->Set('ClearIDCookie','CLEAR');	# Clear User Cookie next time
			$ACTINIC::B2B->Set('ClearUserCookie','CLEAR');	# Clear User Name Cookie next time
			}
		else
			{
			($sDigest,$sBaseFile) = ACTINIC::CaccGetCookies();		# See if the user logged in already
			$ACTINIC::B2B->Set('BaseFile',$sBaseFile);
			}
		}

	my ($Status, $sMessage, $pBuyer) = ACTINIC::GetBuyer($sDigest, ACTINIC::GetPath()); # look up the buyer
	if ($Status != $::SUCCESS &&
		 $Status != $::NOTFOUND)
		{
		#
		# Dup the last entry in the page list to help the bounce
		#
		push (@::g_PageList, $::g_PageList[$#::g_PageList]);
		my ($Status, $sMessage, $sHTML) = ACTINIC::ReturnToLastPage(7, "<FONT SIZE=\"+2\">" . $sMessage . "</FONT>", ACTINIC::GetPhrase(-1, 141),
																						\@::g_PageList, $::g_sWebSiteUrl,
																						$::g_sContentUrl, $::g_pSetupBlob, %::g_InputHash);
		if ($Status != $::SUCCESS)
			{
			ACTINIC::ReportError($sMessage, ACTINIC::GetPath());
			}

		ACTINIC::UpdateDisplay($sHTML, $::g_OriginalInputData, \@::g_PageList);
		exit;
		}

	if( $sDigest &&
		 $Status != $::NOTFOUND)						# Find the user
		{
		my $pAccount;
		($Status, $sMessage, $pAccount) = ACTINIC::GetCustomerAccount($$pBuyer{AccountID}, ACTINIC::GetPath());
		if ($Status != $::SUCCESS)
			{
			#
			# Dup the last entry in the page list to help the bounce
			#
			push (@::g_PageList, $::g_PageList[$#::g_PageList]);
			my ($Status, $sMessage, $sHTML) = ACTINIC::ReturnToLastPage(7, "<FONT SIZE=\"+2\">" . $sMessage . "</FONT>", ACTINIC::GetPhrase(-1, 141),
																						 \@::g_PageList, $::g_sWebSiteUrl,
																						 $::g_sContentUrl, $::g_pSetupBlob, %::g_InputHash);
			if ($Status != $::SUCCESS)
				{
				ACTINIC::ReportError($sMessage, ACTINIC::GetPath());
				}

			ACTINIC::UpdateDisplay($sHTML, $::g_OriginalInputData, \@::g_PageList);
			exit;
			}

		if( $$pAccount{Status} != 0 )					# Customer account suspended
			{
			my ($Status, $sError, $sHTML) = ACTINIC::BounceToPageEnhanced(7, "<FONT SIZE=\"+2\">" . ACTINIC::GetPhrase(-1, 214, $$pAccount{AccountName}) . "</FONT>", 
																										'',
																									  \@::g_PageList, $::g_sWebSiteUrl,
																									  $::g_sContentUrl, $::g_pSetupBlob, 
																									  $::g_sWebSiteUrl, 
																						  			  \%::g_InputHash);
			ACTINIC::PrintPage($sHTML, undef, $::FALSE);
			exit;
			}
		elsif ( $$pBuyer{Status} != 0 )				# Buyer account suspended
			{
			my ($Status, $sError, $sHTML) = ACTINIC::BounceToPageEnhanced(7, "<FONT SIZE=\"+2\">" . ACTINIC::GetPhrase(-1, 215, $$pBuyer{Name},$$pAccount{AccountName}) . "</FONT>", 
																							'',
																						  \@::g_PageList, $::g_sWebSiteUrl,
																						  $::g_sContentUrl, $::g_pSetupBlob, 
																						  $::g_sWebSiteUrl, 
																						  \%::g_InputHash);
			ACTINIC::PrintPage($sHTML, undef, $::FALSE);
			exit;
			}
		$ACTINIC::B2B->Set('UserDigest',$sDigest);

		CaccSetCheckoutFields($pBuyer, $pAccount);
		
		ACTINIC::CloseCustomerAddressIndex(); # The customer index is left open for multiple access, so clean it up here
		}
	else														# Not found
		{
		my $sMessage = ACTINIC::GetPhrase(-1, 216);
		RecordErrors($sMessage, ACTINIC::GetPath()); # record the error to the error file
		
		push @::g_PageList, ACTINIC::GetReferrer();
		my ($Status, $sError, $sHTML) = ACTINIC::ReturnToLastPage(7, "<FONT SIZE=\"+2\">" . $sMessage . "</FONT>", ACTINIC::GetPhrase(-1, 208),
																						\@::g_PageList, $::g_sWebSiteUrl,
																						$::g_sContentUrl, $::g_pSetupBlob, %::g_InputHash);
		if ($Status != $::SUCCESS)
			{
			ACTINIC::ReportError($sError, ACTINIC::GetPath());
			}
	
		PrintPage($sHTML, undef, $::TRUE);
		exit;
		}
	}

#######################################################
#																		
# CAccCatalogBody - returns name of main catalog page 
#
# Params - none
# Returns 0 - html file name - no path
#         1 - the same file if there are no frames and a frameset file if there are
#
#######################################################

sub CAccCatalogBody
	{
	my $sProductPage = 'catalogbody.html';				# default template
	#
	# See if there is a template request
	#
	if( $::g_InputHash{PRODUCTPAGE} =~ /\S/ )
		{
		$sProductPage = $::g_InputHash{PRODUCTPAGE};
		}

	my $sFramePage = $sProductPage;
	if( ACTINIC::IsCatalogFramed() )
		{
		$sFramePage = 'frameset.html';			# default template with frames
		}
	return ($sProductPage,$sFramePage);
	}

#######################################################
#
# CaccSetCheckoutFields - Set the details for this buyer
#			to the checkout fields
#
# Input  0 - ref to buyer
#        1 - ref to account
#
#######################################################

sub CaccSetCheckoutFields
	{
	my ($pBuyer, $pAccount) = @_;
	my ($Status, $sMessage, $pInvoiceAddress, $pDeliveryAddress, $nInvoiceAddressID, $nDeliveryAddressID);
	#
	# Set the address IDs to undefined
	#
	$nInvoiceAddressID = -1;
	$nDeliveryAddressID = -1;

	#
	# Parse the tax information
	#
	ActinicOrder::ParseAdvancedTax();
	#
	# Set the Compnay fields to the account name
	#
	$::g_BillContact{'REMEMBERME'} = $::FALSE;
	#
	# Set the Company fields to the account name
	#
	$::g_BillContact{'COMPANY'} = $pAccount->{AccountName};

	my (%PaymentInfo);
	#
	# Set the preferred payment method after getting the string
	# representation for use in the payment hash
	#
	$PaymentInfo{'METHOD'} 		= ActinicOrder::EnumToPaymentString($pAccount->{DefaultPaymentMethod});
	$PaymentInfo{'SCHEDULE'} 	= $pAccount->{PriceSchedule};				# get the schedule ID;
	#
	# Check if the account only allows one invoice address
	#
	if($pAccount->{InvoiceAddressRule} == 1)
		{
		$nInvoiceAddressID = $pAccount->{InvoiceAddress};

		$::g_BillContact{'NAME'}		= $pAccount->{Name};
		$::g_BillContact{'SALUTATION'}= $pAccount->{Salutation};
		$::g_BillContact{'JOBTITLE'}	= $pAccount->{Title};
		$::g_BillContact{'PHONE'}		= $pAccount->{TelephoneNumber};
		$::g_BillContact{'FAX'}			= $pAccount->{FaxNumber};
		$::g_BillContact{'EMAIL'}		= $pAccount->{EmailAddress};
		}
	else
		{
		if($pBuyer->{InvoiceAddressRule} == 0)
			{
			$nInvoiceAddressID = $pBuyer->{InvoiceAddressID};
			}

		$::g_BillContact{'NAME'}		= $pBuyer->{Name};
		$::g_BillContact{'SALUTATION'}= $pBuyer->{Salutation};
		$::g_BillContact{'JOBTITLE'}	= $pBuyer->{Title};
		$::g_BillContact{'PHONE'}		= $pBuyer->{TelephoneNumber};
		$::g_BillContact{'FAX'}			= $pBuyer->{FaxNumber};
		$::g_BillContact{'EMAIL'}		= $pBuyer->{EmailAddress};
		}
	#
	# If we know the invoice address populate the hashes with the address
	# details
	#
	if($nInvoiceAddressID != -1)
		{
		($Status, $sMessage, $pInvoiceAddress) = 
			ACTINIC::GetCustomerAddress($pBuyer->{AccountID}, $pAccount->{InvoiceAddress}, ACTINIC::GetPath());
		if ($Status != $::SUCCESS)
			{
			return();
			}
		$::g_BillContact{'ADDRESS1'}		= $pInvoiceAddress->{Line1};
		$::g_BillContact{'ADDRESS2'}		= $pInvoiceAddress->{Line2};
		$::g_BillContact{'ADDRESS3'}		= $pInvoiceAddress->{Line3};
		$::g_BillContact{'ADDRESS4'}		= $pInvoiceAddress->{Line4};
		$::g_BillContact{'COUNTRY'}		= ACTINIC::GetCountryName($pInvoiceAddress->{CountryCode});
		$::g_BillContact{'POSTALCODE'}	= $pInvoiceAddress->{PostCode};
		#
		# Now set the invoice location information
		#
		$::g_LocationInfo{INVOICE_COUNTRY_CODE}	= $pInvoiceAddress->{CountryCode};
		$::g_LocationInfo{INVOICE_REGION_CODE}		= $pInvoiceAddress->{StateCode};
		#
		# Now set any tax exemption data
		#
		if($::g_pTaxSetupBlob{TAX_BY} != 2)
			{
			$::g_TaxInfo{'EXEMPT1'} 	= $pInvoiceAddress->{ExemptTax1} == 0 ? $::FALSE : $::TRUE;
			$::g_TaxInfo{'EXEMPT2'} 	= $pInvoiceAddress->{ExemptTax2} == 0 ? $::FALSE : $::TRUE;
			if($::g_TaxInfo{'EXEMPT1'})
				{
				$::g_TaxInfo{'EXEMPT1DATA'} 	= $pInvoiceAddress->{Tax1ExemptData};
				}
			if($::g_TaxInfo{'EXEMPT2'})
				{
				$::g_TaxInfo{'EXEMPT2DATA'} 	= $pInvoiceAddress->{Tax2ExemptData};
				}
			}
		}

	if($pBuyer->{DeliveryAddressRule} == 0)
		{
		$nDeliveryAddressID = $pBuyer->{DeliveryAddressID};
		($Status, $sMessage, $pDeliveryAddress) = 
			ACTINIC::GetCustomerAddress($pBuyer->{AccountID}, $pAccount->{DeliveryAddressID}, ACTINIC::GetPath());

		$::g_ShipContact{'NAME'}		= $pBuyer->{Name};
		$::g_ShipContact{'SALUTATION'}= $pBuyer->{Salutation};
		$::g_ShipContact{'JOBTITLE'}	= $pBuyer->{Title};
		$::g_ShipContact{'PHONE'}		= $pBuyer->{TelephoneNumber};
		$::g_ShipContact{'FAX'}			= $pBuyer->{FaxNumber};
		$::g_ShipContact{'EMAIL'}		= $pBuyer->{EmailAddress};

		$::g_ShipContact{'ADDRESS1'}		= $pDeliveryAddress->{Line1};
		$::g_ShipContact{'ADDRESS2'}		= $pDeliveryAddress->{Line2};
		$::g_ShipContact{'ADDRESS3'}		= $pDeliveryAddress->{Line3};
		$::g_ShipContact{'ADDRESS4'}		= $pDeliveryAddress->{Line4};
		$::g_ShipContact{'COUNTRY'}		= ACTINIC::GetCountryName($pDeliveryAddress->{CountryCode});
		$::g_ShipContact{'POSTALCODE'}	= $pDeliveryAddress->{PostCode};
		#
		# Now set the delivery location information
		#
		$::g_LocationInfo{DELIVERY_COUNTRY_CODE}	= $pDeliveryAddress->{CountryCode};
		$::g_LocationInfo{DELIVERY_REGION_CODE}	= $pDeliveryAddress->{StateCode};
		#
		# Now set any tax exemption data if we're taxing by delivery address
		#
		if($::g_pTaxSetupBlob{TAX_BY} == 2)
			{
			$::g_TaxInfo{'EXEMPT1'} 	= $pDeliveryAddress->{ExemptTax1} == 0 ? $::FALSE : $::TRUE;
			$::g_TaxInfo{'EXEMPT2'} 	= $pDeliveryAddress->{ExemptTax2} == 0 ? $::FALSE : $::TRUE;
			if($::g_TaxInfo{'EXEMPT1'})
				{
				$::g_TaxInfo{'EXEMPT1DATA'} 	= $pDeliveryAddress->{Tax1ExemptData};
				}
			if($::g_TaxInfo{'EXEMPT2'})
				{
				$::g_TaxInfo{'EXEMPT2DATA'} 	= $pDeliveryAddress->{Tax2ExemptData};
				}
			}
		}
	#
	# save the modified data
	#
	my @Response = ActinicOrder::GetCartID(ACTINIC::GetPath()); # retrieve the cart ID
	if ($Response[0] != $::SUCCESS)							# error out
		{
		return (@Response);
		}
	my ($sCartID) = $Response[2];
	@Response = ActinicOrder::SaveCheckoutStatus(ACTINIC::GetPath(), $sCartID, \%::g_BillContact,
										\%::g_ShipContact, \%::g_ShipInfo, \%::g_TaxInfo, \%::g_GeneralInfo,
										\%PaymentInfo, \%::g_LocationInfo);
	}

#######################################################
# CAccFindUser - find logged in user using cookie
# No arguments
# Returns User Digest or "" if not found or suspended
#######################################################

sub CAccFindUser
	{
	my ($sDigest,$sBaseFile) = ACTINIC::CaccGetCookies();	# See if the user logged in already	
	if (!$sDigest)
		{
		return ("");
		}

	my ($Status, $sMessage, $pBuyer) = ACTINIC::GetBuyer($sDigest, ACTINIC::GetPath()); # look up the buyer
	if ($Status != $::SUCCESS)
		 {
		 return ("");
		 }

	my $pAccount;
	($Status, $sMessage, $pAccount) = ACTINIC::GetCustomerAccount($$pBuyer{AccountID}, ACTINIC::GetPath());
	if ($Status != $::SUCCESS)
		{
		return ("");
		}

	if( $$pAccount{Status} == 0 &&
		 $$pBuyer{Status} == 0 )						# Check if account is active
		{
		$ACTINIC::B2B->Set('BaseFile',$sBaseFile);
		return ($sDigest);								# Found - return digest
		}

	return ("");
	}

#######################################################
# ParseXML - PXML wrapper
# Prepares basic variables and parses text using ACTINIC_PXML
# Argument: - text to parse
# Returns: - Parsed text
#######################################################

sub ParseXML
	{
	my $sHTML = shift;
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');

	if( !$sDigest )	# No user
		{
		$sDigest = $ACTINIC::B2B->Set('UserDigest',ACTINIC::CAccFindUser());	# See if there is a user cookie after all
		}

	if( $sDigest )		# User found - do some basic XML variables
		{
		my ($Status, $sMessage, $pBuyer) = ACTINIC::GetBuyer($sDigest, ACTINIC::GetPath()); # look up the buyer
		if ($Status != $::SUCCESS)
			{
			ACTINIC::ReportError($sMessage, ACTINIC::GetPath());
			}

		my $pAccount;
		($Status, $sMessage, $pAccount) = ACTINIC::GetCustomerAccount($$pBuyer{AccountID}, ACTINIC::GetPath());
		if ($Status != $::SUCCESS)
			{
			ACTINIC::ReportError($sMessage, ACTINIC::GetPath());
			}

		my $sBuyer = $$pBuyer{Name};
		my $sAccount = $$pAccount{AccountName};
		$ACTINIC::B2B->SetXML('BUYER',      $sBuyer);
		$ACTINIC::B2B->SetXML('ACCOUNT',    $sAccount);
		$ACTINIC::B2B->SetXML('NOWSERVING', ACTINIC::GetPhrase(-1, 212, $sBuyer));
		$ACTINIC::B2B->SetXML('CURRACCOUNT',ACTINIC::GetPhrase(-1, 213, $sAccount) . "<Actinic:LOGOUT_SIMPLE/>");
		$ACTINIC::B2B->SetXML('WELCOME',    ACTINIC::GetPhrase(-1, 210, $$::g_pSetupBlob{FORM_EMPHASIS_COLOR}, $$::g_pSetupBlob{FORM_BACKGROUND_COLOR},$sBuyer));

		my $sShop = $::g_InputHash{SHOP} ? '&SHOP=' . ACTINIC::EncodeText2($::g_InputHash{SHOP}, $::FALSE) : '';
		$ACTINIC::B2B->SetXML('LOGOUT', "</TR><TR><TD ALIGN=RIGHT>"
									 . "<A HREF=\"$::g_sAccountScript\?ACTION=LOGOUT" 
									 . $sShop 
									 . '" TARGET="_parent">'
									 . "<B>" 
									 . ACTINIC::GetPhrase(-1, 217) 
									 . "</B></A></TD>");
		$ACTINIC::B2B->SetXML('LOGOUT_SIMPLE', 
									 "&nbsp;<A HREF=\"$::g_sAccountScript\?ACTION=LOGOUT&PATH=$::g_InputHash{PATH}" . 
									 $sShop 
									 . '" TARGET="_parent">'
									 . ACTINIC::GetPhrase(-1, 217) 
									 . "</A>");
		}

	my $pXML = new ACTINIC_PXML();						# Create XML object
	my $sParsedHTML = $pXML->Parse($sHTML);			# Parse text
	if ($$::g_pSetupBlob{USE_SSL})						# we are using SSL security
		{
		if ( $sDigest )
			{
			$sParsedHTML =~ s/http:\/\//https:\/\//gi;	# make everything use secure transfer for Business
			}
		else
			{
			$sParsedHTML =~ s/src\s*=\s*http:\/\//src=https:\/\//gi;	# make the images, etc. use secure transfer
			}
		}

	return ($sParsedHTML);		# Return parsed text
	}
############################################################
#  package ACTINIC_B2B - keeps B2B variables
#  This object keeps B2B variables providing Set, Clear and Get
#  functions.
#  SetXML, AppendXML, GetXML and ClearXML are used to store and
#  retrieve XML tag variables used by ACTINIC_PXML class.
#
#  Ryszard Zybert  Mar 17 12:11:17 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################
package ACTINIC_B2B;
use strict;
############################################################
#  sub new - create B2B object 
#
#  Ryszard Zybert  Mar 17 12:11:50 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub new 
	{
	my $Proto = shift;
	my $Class = ref($Proto) || $Proto;
	my $Self  = {};
	bless ($Self, $Class);
	$Self->{XML} = {};

	return $Self;
	}
############################################################
#  B2B->Set - set B2B variable
#  Arguments
#		0 - class
#		1 - variable name
#		2 - variable value
#  Returns
#		0 - variable value
#  
#  Ryszard Zybert  Mar 17 12:14:43 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub Set
	{
	my $Self = shift;
	my $sName = shift;
	my $sValue = shift;

	$Self->{$sName} = $sValue;
	return $sValue;
	}
############################################################
#  B2B->Clear - unset B2B variable
#  Arguments
#  		0 - class
#  		1 - variable name
#
#  Ryszard Zybert  Mar 17 12:19:09 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub Clear
	{
	my $Self = shift;
	my $sName = shift;

	$Self->{$sName} = undef;
	}
############################################################
#  B2B->Get - get B2B variable
#  Arguments
#    		0 - class
#    		1 - variable name
#  Returns
#    		0 - variable value
#
#  Ryszard Zybert  Mar 17 12:20:48 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub Get
	{
	my $Self = shift;
	my $sName = shift;
	return $Self->{$sName};
	}
############################################################
#  B2B->SetXML - set B2B XML variable
#  Arguments
#		0 - class
#		1 - variable name
#		2 - variable value
#  Returns
#		0 - variable value
#  
#  If variable is already defined does nothing and returns an empty
#  string.
#  Update should be used to modify existing variables
#
#  Ryszard Zybert  Mar 17 12:14:43 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub SetXML
	{
	my $Self = shift;
	my $sName = shift;
	my $sValue = shift;

	$Self->{XML}->{$sName} = $sValue;
	return $sValue;
	}
############################################################
#  B2B->AppendXML - append string to B2B XML variable
#  Arguments
#  		0 - class
#  		1 - variable name
#  		2 - string to append
#  Returns
#  		0 - new variable value
#
#  Ryszard Zybert  Mar 17 12:19:09 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub AppendXML
	{
	my $Self = shift;
	my $sName = shift;
	my $sValue = shift;

	$Self->{XML}->{$sName} .= $sValue;
	return $Self->{XML}->{$sName};
	}
############################################################
#  B2B->GetXML - get B2B XML variable
#  Arguments
#    		0 - class
#    		1 - variable name
#  Returns
#    		0 - variable value
#
#  Ryszard Zybert  Mar 17 12:20:48 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub GetXML
	{
	my $Self = shift;
	my $sName = shift;
	return $Self->{XML}->{$sName};
	}
############################################################
#  B2B->ClearXML - clear all B2B XML variables
#
#  Ryszard Zybert  Mar 17 12:23:28 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub ClearXML
	{
	my $Self = shift;
	$Self->{XML} = undef;
	}

############################################################
#  PXML.pm - pseudo XML parser
#
#  Ryszard Zybert  Nov 28 09:30:40 GMT 1999
#
#  Copyright (c) Actinic Software Ltd 1999
############################################################
package PXML;
use strict;
############################################################
#  PXML->new() - constructor for PXML class
#  A very standard constructor. Allows inheritance.
#  Calls Set() function passing it all the arguments.
#  So the arguments may be specified here with name=>value
#  pairs or they may be set later using Set() method.
#  No arguments are obligatory in new() but all arguments must
#  be specified before Parse() method is used.
#  Following arguments are required:
#  ID       => prefix of tags to be handled (it may be specified in Parse())
#  tag1     => reference to function to handle <IDtag1>
#  tag1_End => reference to function to handle </IDtag1>
#  ...
#  Special optional arguments:
#  DEFAULT  => reference to a function handling unrecognised
#              tags (with prefix specified in ID)
#  XMLERROR => Error message to print when parser detects an error
#              Embedded %s will be replaced by the tag for which an error
#              was detected.
#              Default: "Error parsing XML text (%s)"
#  If DEFAULT is not specified the unknown tags are passed to
#  output unchanged.
#
#  Supplied function may or may not be different, the tag
#  name and the ID are passed to the function.
#  See comment for Parse().
#
#  Ryszard Zybert  Dec  1 18:15:36 GMT 1999
#
#  Copyright (c) Actinic Software Ltd 1999
############################################################

sub new 
	{
	my $Proto = shift;
	my $Class = ref($Proto) || $Proto;
	my $Self  = {};
	bless ($Self, $Class);
	$Self->{XMLERROR} = "Error parsing XML text (%s)";
	$Self->{LoopProtect} = 25000;
	$Self->{CurrentLoop} = 0;
	$Self->Set(@_);
#	use constant Version => '4';
	return $Self;
	}
############################################################
#  PXML->Set() - set configuration parameters
#
#  Ryszard Zybert  Nov 28 09:34:32 GMT 1999
#
#  Copyright (c) Actinic Software Ltd 1999
############################################################

sub Set
	{
	my $Self       = shift;
	my %Parameters = @_;
	#
	# Separate handlers from parameters amd make a hash
	#
	foreach (keys %Parameters)
		{
		if( ref($Parameters{$_}) eq "CODE" )	# Treat all functions as tag handlers (case sensitive)
			{
			$Self->{Tags}->{uc($_)} = $Parameters{$_};
			}
		else												# Anything else is a parameter (case sensitive)
			{
			$Self->{$_} = $Parameters{$_};
			}
		}
 	}
############################################################
#  PXML->Parse() - parse text
#  
#  NOTE: It may be called recursively
#
#   Arguments:	0 text to parse
#					1 (optional) ID - prefix to look for
#   Returns:   parsed text
#  
#  When a tag is found, looks for the end-tag and calls function
#  which was declared to deal with this tag.
#  The text between start-tag and end-tag is parsed recursively
#  Then (if defined) function handing $tag.'_End' is called.
#  Abbreviated syntax: <tag/> is accepted.
#  Parameters are parsed and passed as a hash reference to the
#  handler (parameter without value is set to 'SET')
#
#  Tag handling function is called with five arguments:
#		$tag				- tag name
#		\$sText			- reference to text found between start and end tag
#		\%Parameters	- reference to parameter hash
#		ID					- prefix for this run
#		$sStartTag		- full text of start tag
#
#  It may return text to go to output and may also modify the
#  text between tags before it is parsed further
#
#  If end-tag handling function is defined then it is called
#  after the contents is parsed.
#  If not - it defaults to tag handling function.
#  End-tag handling function is called with the same arguments
#  as tag handling function but only arg 0, 3 and 4 are set.
#
#  Tags and ID are case sensitive
#  Parameters names are used in hash unchanged
#
#  If the tag handling function is not defined then:
#  If DEFAULT function is defined - it is called
#  If DEFAULT_End function is defined - it is called for End-tag
#  Otherwise the the tag is passed on to output unchanged.
#
#  Ryszard Zybert  Nov 28 14:40:12 GMT 1999
#
#  Copyright (c) Actinic Software Ltd 1999
############################################################

sub Parse
	{
	my $Self  = shift;
	my $sText = shift;
	my $sId   = shift;

	if( !$sId ) { $sId = $Self->{ID}; }				# ID parameter is optional here
	my $Result;

	$Self->{CurrentLoop}++;
	if( $Self->{CurrentLoop} > $Self->{LoopProtect} )
		{
		$Result = $Self->{XMLERROR};
		$Result =~ s/\%s/Infinite Loop \(\?\)/;
		return $Result;
		}
	#
	# This is where parsing is done
	# To change what is and what is not accepted/processed change the
	# regular expression below - note that results are used below, so
	# if brackets are added/removed, make sure that following code gets
	# fixed.
	#
	while ( $sText =~ / 
			  (								# Start tag 										($1)
				< 
				\s*							# Possible white space at the beginning
				$sId							# Identifier
				([0-9a-zA-Z_]+?)			# Tag	name 											($2)(used)
				(								# Optional parameter list 						($3)(used)
				 (\s+							# Parameter starts from space					($4)
				  [0-9a-zA-Z_]+?			# Parameter name
				  (\=							# Parameter value with equal sign			($5)
					(							# Parameter value									($6)
					 (\"[^\"]+\") |		# Parameter value in double quotes 			($7)
					 (\'[^\']+\') |		# Parameter value in single quotes 			($8)
					 ([^\"\'\ \/\>]+)		# Parameter value without quotes 			($9)
					)	
				  )*?							# Value is optional (default is 'SET')
				 )*?							# Parameters are optional
				) 
				\s*							# Possible white space
				(\/*?)						# Optional End mark 								($10)(used)
				\s*							# Possible white space at the end
				>
			  )
			  | (<!--.*?-->)				# Or comment 										($11)(used)
			  /sx ) 
		{
		$sText   = $';						# shift the buffer pointer
		$Result .= $`;						# add the front part

		#
		# Commented text is not processed
		# Usually this just saves time. But - note that if handlers have side effects
		# then it may matter a bit more if you comment out XML tags
		# To change that - simply comment out comment detection line from regexp above
		#
		if( $11 )							# If comment - pass it as is
			{
			$Result .= $&;					# Add commented part and continue
			next; 
			}
		
		my $sTag					= $2;			# tag name
		my $sParameterText	= $3;			# parameter list
		my $sInsideText		= "";			# text between start and end (empty for now)
		my $sStartTag			= $&;			# complete start tag
		my $sEndTag;							# complete end tag (nothing yet)
		my $ParameterHash;					# hash of parameters (nothing yet)

		#
		# If there are parameters make a hash
		#
		if( $sParameterText )
			{ 
			$ParameterHash = $Self->ParseParameters($sParameterText);
			}
		
		#
		# If not 'abbreviated syntax' look for end-tag
		#
		if( !$10 ) { $sInsideText  = $Self->FindEndTag($sId,$sTag,\$sText,\$sEndTag); }
		
		#
		# If tag handler or DEFAULT defined, call it, otherwise just return the whole text
		# In any case, parse the text recursively
		# 
		my $sGeneralTag = uc($sTag);
		if( !defined($Self->{Tags}->{$sGeneralTag}) ) { $sGeneralTag = 'DEFAULT' }

		if( defined($Self->{Tags}->{$sGeneralTag}) )			# Tag handler found
			{ 
			#
			# Call tag handler and parse text that it returns
			#
			my $sReplace =	&{$Self->{Tags}->{$sGeneralTag}}(
																		  $sTag,					# Tag name
																		  \$sInsideText,		# Reference to text between tags
																		  $ParameterHash,		# Reference to hash of parameters
																		  $sId,					# Current Prefix
																		  $sStartTag			# Full text of start tag
																		 );

			if( $sReplace eq $sStartTag )							# Try to avoid infinite loops
				{															# If nothing changed, don't parse again
				$Result .= $sReplace;
				}
			else
				{
				$Result .= $Self->Parse($sReplace,$sId);
				}
			#
			# Parse text between start-tag and end-tag
			#
			$Result .= $Self->Parse($sInsideText,$sId);
			
			if( defined($Self->{Tags}->{$sGeneralTag.'_END'}) )	# End-tag - call handler and parse returned text
				{
				$sReplace = &{$Self->{Tags}->{$sGeneralTag.'_END'}}('/'.$sTag, "", "", $sId, $sEndTag);
				}
			else																	# Default to the same as start tag
				{
				$sReplace = &{$Self->{Tags}->{$sGeneralTag}}('/'.$sTag, "", "", $sId, $sEndTag);
				}

			if( $sReplace eq $sEndTag )									# Try to avoid infinite loops
				{																	# If nothing changed, don't parse again
				$Result .= $sReplace;
				}
			else
				{
				$Result .= $Self->Parse($sReplace,$sId);
				}
			}
		else												# No handler and no default, just parse text between tags
			{	
			$Result .= $sStartTag . $Self->Parse($sInsideText,$sId) . $sEndTag;
			}
		}
	return $Result . $sText		# Append all the rest of text if no more tags
	}
############################################################
#  PXML->FindEndTag() - find end-tag
#   Arguments:    $sId - current ID
#                 $sTag - tag name
#                 \$sText - reference to text to look in
#                 \$sEnd - reference to end tag (initialy empty)
#   $sText is changed to start after the end-tag
#   Returns: text found before the end-tag
#
#  Ryszard Zybert  Nov 28 14:42:23 GMT 1999
#
#  Copyright (c) Actinic Software Ltd 1999
############################################################

sub FindEndTag
	{
	my $Self = shift;
	my ($sId, $sTag, $sText, $sEnd) = @_;
	
	if( $$sText =~ / < \s* \/ $sId $sTag \s* > /sx )		# Look for end-tag
		{
		$$sText = $';					# Text after end tag
		$$sEnd  = $&;					# Text of end tag
		return $`;						# Text between start-tag and end-tag
		}
	else									# Not found - return error and unchanged text
		{
		my $sErr = $Self->{XMLERROR};
		$sErr    =~ s/\%s/$sId$sTag/;
		return $sErr . $$sText;
		}
	}
############################################################
#  PXML->ParseParameters() - parse parameter list
#  Splits parameter list and makes a hash
#  
#   Arguments:	parameter string (must start with white space)
#   Returns:	parameter hash reference
#
#  Ryszard Zybert  Nov 30 10:47:24 GMT 1999
#
#  Copyright (c) Actinic Software Ltd 1999
############################################################

sub ParseParameters
	{
	my $Self        = shift;
	my $sParameters = shift;
	
	my $ParameterHash = ();
	
	#
	# IMPORTANT:
	# Parameter string starts IMMEDIATELY after recognised TAG
	# So: it MUST start from white space
	#
	while ( $sParameters =~ m/\G 
			  \s+								# Obligatory white space
			  ([0-9a-zA-Z_]+)				# Parameter name ($1)
			  (\=
				(	
				 (\"[^\"]+\") |			# Parameter value in double quotes
				 (\'[^\']+\') |			# Parameter value in single quotes
				 ([^\"\'\ \/\>]+)			# Parameter value without quotes
				)								# Parameter value ($3)
			  )*								# '=value' may not be there ($2)
			  /gsx )
		{
		my $sName = $1;
		if( $2 )								# There is a value
			{
			my $sValue = $3;
			$sValue =~ s/^(\"|\')//;	# Remove leading quote
			$sValue =~ s/(\"|\')$//;	# Remove trailing quote
			$ParameterHash->{$sName} = $sValue;
			}
		else									# No value, set it to 'SET'
			{
			$ParameterHash->{$sName} = 'SET';
			}
		}
	return $ParameterHash;
	}

############################################################
#  package ACTINIC_PXML - ACTINIC specific parser
#  This is a protype for development
#
#  Ryszard Zybert  Dec  7 20:52:23 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
############################################################

package ACTINIC_PXML;
#use constant Version => "1.0, (PXML: " . PXML->Version . ")";

use vars qw(@ISA);
@ISA = qw(PXML);

sub new 
	{
	my $Proto = shift;
	my $Class = ref($Proto) || $Proto;
	my $self  = $Class->SUPER::new();														# dont pass arguments, we can use Set()
	
	bless ($self, $Class);
	
	$self->Set(
				  ID						=>	'Actinic:',													# default prefix
				  MAINFRAME				=>	sub { $self->MainFrameTagHandler(@_)		},		# handle url of main frame
				  PRICES					=> sub { $self->PriceTagHandler(@_)				},		# price tag
				  RETAIL_PRICE_TEXT	=>	sub { $self->RetailPriceTextTagHandler(@_)},		# retail price text tag
				  VAR						=> sub { $self->VarTagHandler(@_)				},		# var tag
				  SECTION				=> sub { $self->SectionTagHandler(@_)			},		# section tag
				  ADDRESSES				=> sub { $self->AddressTagHandler(@_)			},		# addresses tag
				  UNREG					=> sub { $self->UnregTagHandler(@_)				},		# unregistered user tag
				  IGNORE					=> sub { $self->IgnoreTagHandler(@_)			},		# IGNORE tag (deletes text)
				  NOTINB2B				=> sub { $self->NotInB2BTagHandler(@_)			},		# NOTINB2B tag (deletes text)
				  DEFAULT				=> sub { $self->DefaultTagHandler(@_)			},		# unknown tags here
				  XMLERROR				=> "<br><font size=+2 color=red><b>". ACTINIC::GetPhrase(-1, 218) . "</b></font><br>",
				  RETAIL_ONLY_SEARCH => sub { $self->RetailOnlySearchTagHandler(@_)},
				  INSERTVAR				=> sub { $self->InsertVarTagHandler(@_)		},		# Insert variable handler
				 );
	$self->Set(@_);
	return $self;
	}
	
############################################################
#  sub InsertVarTagHandler - lookup for the variable and insert
#  	it if defined (else leave it as is)
#		created for v4.0.8 compatibility issues
#  
#  Arguments : $sTag - tag name
#             $sInsideText - reference to text between start and end,
#             $ParameterHash - hash of parameters,
#             $sId - current tag prefix,
#             $sFullTag - full text of current tag;
#  Returns   : replacement for the tag
#
#  Zoltan Magyar  March 29 2001
#
#  Copyright (c) Actinic Software Ltd (2001)
############################################################

sub InsertVarTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	my $sXMLTag = '';
	
	if( $sTag !~ /^\// )									# Ignore end-tags
		{
		#
		# Check for variable to insert
		#
		my $sVar = $ParameterHash->{TYPE};
		if (defined $Self->{Variables}->{$sVar})
			{
			$sXMLTag = $Self->{Variables}->{$sVar};
			$$sInsideText = ""; 
			}
		}
	return $sXMLTag;	
	}

############################################################
#
#  RetailOnlySearchTagHandler - process the search tag
#
#  Note: <Actinic:Actinic:RETAIL_ONLY_SEARCH/> acts globally
#
#  Arguments : $sTag - tag name
#              $sInsideText - reference to text between start and end,
#              $ParameterHash - hash of parameters,
#              $sId - current tag prefix,
#              $sFullTag - full text of current tag;
#
#  Returns   : replacement for the tag
#
############################################################

sub RetailOnlySearchTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	my $sDigest = $ACTINIC::B2B->Get('UserDigest');	# get the user identifying digest
	if ($sTag !~ /^\//)									# If not End tag
		{
		if ($sDigest)										# If there is a user remove tags and text
			{
			if (ref($sInsideText))
				{
				$$sInsideText = "";
				}
			}
		}
	else														# here on the second call (the end tag)
		{
		return ('');										# on the second pass, return nothing to prevent duplicate entries
		}
	#
	# If the buyer exists and does not see retail prices, note that the search is retail prices only
	#
	my $sRetailMessage = ACTINIC::GetPhrase(-1, 357);

	if ($sDigest)											# see which price schedule they use
		{
		my ($Status, $sMessage, $pBuyer) = ACTINIC::GetBuyer($sDigest, ACTINIC::GetPath()); # look up the buyer
		if ($Status == $::SUCCESS)
			{
			my $pAccount;
			($Status, $sMessage, $pAccount) = ACTINIC::GetCustomerAccount($pBuyer->{AccountID}, ACTINIC::GetPath()); # find the account information
			if ($Status == $::SUCCESS)
				{
				if ($pAccount->{PriceSchedule} == $ActinicOrder::RETAILID) # if we can confirm that this buyer is using the retail schedule
					{
					$sRetailMessage = '';				# the message is unnecessary
					}
				}
			}
		}

	return ($sRetailMessage);
	}

############################################################
#  AddressTagHandler - callback for addresses
#  Replaces <Actinic:ADDRESSES/> tag by address table
#  
#  Arguments : $sTag - tag name
#              $sInsideText - reference to text between start and end,
#              $ParameterHash - hash of parameters,
#              $sId - current tag prefix,
#              $sFullTag - full text of current tag;
#  Returns   : address table
#
#  Ryszard Zybert  Jan  3 16:44:37 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub AddressTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');

	if( $sTag !~ /^\// )																				# Ignore end-tags
		{
		my ($Status, $sMessage, $pBuyer) = ACTINIC::GetBuyer($sDigest, ACTINIC::GetPath());
		if ($Status != $::SUCCESS)
			{
			return ("");
			}
		my $pAccount;
		($Status, $sMessage, $pAccount) = ACTINIC::GetCustomerAccount($$pBuyer{AccountID}, ACTINIC::GetPath());
		if ($Status != $::SUCCESS)
			{
			return ("");
			}
		my @AddressIdList = split(/,/, $$pAccount{AddressList});
		my $AddressID;
		my %AddressList;
		foreach $AddressID (@AddressIdList)
			{
			($Status, $sMessage, $AddressList{$AddressID}) = ACTINIC::GetCustomerAddress($$pBuyer{AccountID}, $AddressID, ACTINIC::GetPath());
			if ($Status != $::SUCCESS)
				{
				ACTINIC::CloseCustomerAddressIndex(); # The customer index is left open for multiple access, so clean it up here
				return ("");
				}
			}

		my @Temp = keys %AddressList;
		if ($#Temp == -1)
			{
			ACTINIC::CloseCustomerAddressIndex();		# The customer index is left open for multiple access, so clean it up here
			return ("");
			}

		my ($sType,$sSelect,$nRule,$sChecked);
		if( $ParameterHash->{TYPE} =~ /^INVOICE/ )											# Invoice address
			{
			$sType   = 'ValidAsInvoiceAddress';													# This will be tested for each address
			if( $pAccount->{InvoiceAddressRule} == 1 )										# The Customer rule overrides buyer rule
				{
				$nRule = 0;
				$sSelect = $pAccount->{InvoiceAddress};                               # Default (or fixed) address
				($Status, $sMessage, $AddressList{$sSelect}) = ACTINIC::GetCustomerAddress($$pBuyer{AccountID}, $sSelect, ACTINIC::GetPath());
				if ($Status != $::SUCCESS)	
					{
					ACTINIC::CloseCustomerAddressIndex(); # The customer index is left open for multiple access, so clean it up here
					return ("");
					}
				}
			else
				{
				$nRule   = $pBuyer->{InvoiceAddressRule};										# Address rule for this user
				$sSelect = $pBuyer->{InvoiceAddressID};                              # Default (or fixed) address
				}
			}
		elsif( $ParameterHash->{TYPE} =~ /^DELIVERY/ )										# Delivery address
			{
			$sType   = 'ValidAsDeliveryAddress';												# This will be tested for each address
			$nRule   = $pBuyer->{DeliveryAddressRule};										# Address rule for this user
			$sSelect = $pBuyer->{DeliveryAddressID};											# Default (or fixed) address
			}
		if( $ParameterHash->{TYPE} =~ /FORM$/ )												# Address form
			{
			if( $nRule != 2 )																			# Only shown for Rule 2
				{
				$$sInsideText = "";
				}
			ACTINIC::CloseCustomerAddressIndex();		# The customer index is left open for multiple access, so clean it up here
			return "";
			}

		ACTINIC::CloseCustomerAddressIndex();		# The customer index is left open for multiple access, so clean it up here

		#
		# Format 'mini templates'.
		# Table title and address format depend on $nRule, there must be one for each rule
		#
		my $sTableFormat   	= $Self->{Variables}->{ADDRESS_TABLE};
		my $sTitle         	= $Self->{Variables}->{'ADDRESS_TITLE' . $nRule};
		my $sTitle_1        	= $Self->{Variables}->{'ADDRESS_TITLE1' . $nRule};
		my $sForm				= '<TD>' . $Self->{Variables}->{'ADDRESS_FORM' . $nRule} . '</TD>';
		#
		# Number of columns only matters if it results in more than one row
		# Otherwise existing addresses will expand to fill the table
		#
		my $nColumns         = $Self->{Variables}->{ADDRESS_COLUMNS} || 1;			# Number of columns - default to 1

		if( !$sForm or !$sTableFormat ) 
			{ 
			return "";
			} 										# No formats - we cannot do that
		
		my $sAddressText = "";
		
		if( $nRule == 0 )																				# Rule 0 - fixed address
			{
			$sAddressText .= '<TR><TD>';															# Just a single cell
			$sAddressText .= sprintf($sForm,
											 $sSelect,													# Address ID
											 $AddressList{$sSelect}->{Name},					# Address text follows
											 $AddressList{$sSelect}->{Line1},
											 $AddressList{$sSelect}->{Line2},
											 $AddressList{$sSelect}->{Line3},
											 $AddressList{$sSelect}->{Line4},
											 $AddressList{$sSelect}->{PostCode},
											 ACTINIC::GetCountryName($AddressList{$sSelect}->{CountryCode}));
			$sAddressText .= '</TD></TR>';
			}
		else																								# Rule 1 - select from list
			{																								# Rule 2 - select or fill form
			$sTitle = sprintf($sTitle,ACTINIC::GetPhrase(-1, 302));						# Insert title from prompts
			if( $nRule == 2 )																			# Insert text for address form
				{
				if( $ParameterHash->{TYPE} =~ /^INVOICE/ )
					{
					$sTitle_1 = sprintf($sTitle_1,ACTINIC::GetPhrase(-1, 303,ACTINIC::GetPhrase(-1, 304)));
					}
				else
					{
					$sTitle_1 = sprintf($sTitle_1,ACTINIC::GetPhrase(-1, 303,ACTINIC::GetPhrase(-1, 305)));
					}
				}
			my $nCount = 0;
			my $nRowCount = 0;
			my $sCh;
			foreach (keys %AddressList)
				{
				if( $AddressList{$_}->{$sType} )
					{
					if( $nCount % $nColumns == 0 )
						{
						$sAddressText .= '<TR>';													# New row
						}
					if( $_ eq $sSelect and $nRule == 1 )										# For Rule 1 check default address
						{
						$sCh = ' CHECKED';
						}
					else
						{
						$sCh = '';
						}

					$sAddressText .= sprintf($sForm,
													 ACTINIC::GetPhrase(-1, 301),
													 $_,													# Address ID (for RADIO button)
													 $sCh,												# Optional 'CHECKED'
													 $AddressList{$_}->{Name},					# Address text follows
													 $AddressList{$_}->{Line1},
													 $AddressList{$_}->{Line2},
													 $AddressList{$_}->{Line3},
													 $AddressList{$_}->{Line4},
													 $AddressList{$_}->{PostCode},
													 ACTINIC::GetCountryName($AddressList{$_}->{CountryCode}));
					
					$nCount++;																			# Count cells
					if( $nCount % $nColumns == 0 )												# Full row
						{
						$sAddressText .= '</TR>';													# Close row
						$nRowCount++;																	# Count rows
						}
					}
				}

			while( $nCount % $nColumns != 0 )													# Close table row if not closed
				{
				if( $nRowCount > 0 ) { $sAddressText .= '<TD>&nbsp;</TD>' }				# If more than one row - add empty cells
				$nCount++;
				if( $nCount % $nColumns == 0 )
					{
					$sAddressText .= '</TR>';
					last;
					}
				}
			}
		$sAddressText =~ s/<br>[,\s]*/<br>/gi;													# Remove leading commas
		$sAddressText =~ s/[,\s]*<br>/<br>/gi;													# Remove trailing commas
		return sprintf($sTableFormat,
							$sTitle,
							$$::g_pSetupBlob{FORM_EMPHASIS_COLOR}, 							# Border
							$$::g_pSetupBlob{FORM_BACKGROUND_COLOR}, 							# Background
							$sAddressText,
							$sTitle_1);
		}
	return "";
	}


############################################################
#  sub VarTagHandler - callback for variables
#  Sets variables
#  There should be NAME and VALUE parameters
#  $Self->{Variables}->{name} is set to value
#  
#  Arguments : $sTag - tag name
#             $sInsideText - reference to text between start and end,
#             $ParameterHash - hash of parameters,
#             $sId - current tag prefix,
#             $sFullTag - full text of current tag;
#  Returns   : empty string
#
#  Ryszard Zybert  Dec  7 20:58:25 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
############################################################

sub VarTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )
		{
		$Self->{Variables}->{$ParameterHash->{NAME}} = $ParameterHash->{VALUE};
		}
	return "";
	}

############################################################
#  sub DefaultTagHandler - callback for unknown tags
#  Looks up the tag in the B2B XML Tags hash
#  If found - replaces the tag by it
#  (meaning the whole <Actinic:$sTag....> sequence)
#  If not doesn't -leaves everything untouched
#  
#  Arguments : $sTag - tag name
#             $sInsideText - reference to text between start and end,
#             $ParameterHash - hash of parameters,
#             $sId - current tag prefix,
#             $sFullTag - full text of current tag;
#  Returns   : replacement for the tag
#
#  Ryszard Zybert  Dec  7 20:58:25 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
############################################################

sub DefaultTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	my $sXMLTag = $ACTINIC::B2B->GetXML($sTag);
	return (defined($sXMLTag)) ? $sXMLTag : $sFullTag;	
	}

############################################################
#
#  sub RetailPriceTextTagHandler - callback for retail text
#  Sets XML variable
#  B2B->{XML}->{tag} is set to value of text between tags
#  
#  Arguments : $sTag - tag name
#             $sInsideText - reference to text between start and end,
#             $ParameterHash - hash of parameters,
#             $sId - current tag prefix,
#             $sFullTag - full text of current tag;
#  Returns   : empty string
#
############################################################


sub RetailPriceTextTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )											# Ignore end-tag completely
		{
		if(ref($sInsideText))									# If there is text store it in XML variable
			{
			$ACTINIC::B2B->SetXML($sTag, $$sInsideText);
			$$sInsideText = "";									# Text not needed anymore
			}
		}
	return "";														# Both tags also removed
	}

############################################################
#  DefaultRemovingTagHandler - callback for unknown tags
#  As DefaultTagHandler except that unknown tags are removed
#  together with all text between tags
#
#  Looks up the tag in the B2B XML Tags hash
#  If found - replaces the tag by it
#  (meaning the whole <Actinic:$sTag....> sequence)
#  If not removes everything
#  
#  Arguments : $sTag - tag name
#              $sInsideText - reference to text between start and end,
#              $ParameterHash - hash of parameters,
#              $sId - current tag prefix,
#              $sFullTag - full text of current tag;
#  Returns   : replacement for the tag
#
#  Ryszard Zybert  Dec  7 20:58:25 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
############################################################

sub DefaultRemovingTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	my $sXMLTag = $ACTINIC::B2B->GetXML($sTag);
	if( defined($sXMLTag) )	# Don't touch text, return replacement tag
		{
		return $sXMLTag;
		}
	else											# Clear both text and tag
		{
		if( ref($sInsideText) ) { $$sInsideText = ""; }
		return "";
		}
	}

############################################################
#  IgnoreTagHandler
#  Remove text within the tag and the tag
#
#  Ryszard Zybert  Jul 25 22:30:51 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub IgnoreTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	if( ref($sInsideText) ) { $$sInsideText = ""; }
	return "";
	}

############################################################
#  NotInB2BTagHandler
#  If there is a registered user removes the tags end text between
#    Arguments : $sTag - tag name
#                $sInsideText - reference to text between start and end,
#                $ParameterHash - hash of parameters,
#                $sId - current tag prefix,
#                $sFullTag - full text of current tag;
#    Returns   : ""
#
#  Ryszard Zybert  Jul 28 14:20:06 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub NotInB2BTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if( $sTag !~ /^\// )																						# If not End tag
		{
		my $sDigest = $ACTINIC::B2B->Get('UserDigest');
		if( $sDigest )			# If there is a user remove tags and text
			{
			if( ref($sInsideText) ) { $$sInsideText = ""; }
			}
		}
	return "";
	}

############################################################
#  UnregTagHandler - callback for UNREG tag
#  If there is a registered user removes the tags end text between
#  Otherwise produces a warning page and bounces to login page
#
#  Arguments : $sTag - tag name
#              $sInsideText - reference to text between start and end,
#              $ParameterHash - hash of parameters,
#              $sId - current tag prefix,
#              $sFullTag - full text of current tag;
#  Returns   : replacement for the tag
#
#  Ryszard Zybert  May 16 15:23:37 BST 2000
#
#  Copyright (c) Actinic Software Ltd (1999)
############################################################

sub UnregTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');
	
	if( $sTag !~ /^\// )																						# If not End tag
		{
		if( $sDigest )			# If there is a user remove tags and text
			{
			if( ref($sInsideText) ) { $$sInsideText = ""; }
			return "";
			}
		else
			{
			#
			# This will only have effect if JavaScript is disabled.
			# Otherwise there is a JavaScript alert and this script is no called
			# So - in this case we just show warning and jup back to original page
			#
			my $sReferer =  ACTINIC::GetReferrer();					# Current page
			$sReferer =~ s"/[^/]*$"/";										# Remove file name
			push @::g_PageList, $sReferer;								# Store index page
			push @::g_PageList, $sReferer;								# Store index page (twice)

			my ($Status, $sError, $sHTML) = ACTINIC::ReturnToLastPage(7," " , 
																						 ACTINIC::GetPhrase(-1, 208),
																						\@::g_PageList, $::g_sWebSiteUrl,
																						$::g_sContentUrl, $::g_pSetupBlob, %::g_InputHash);
			if ($Status != $::SUCCESS)		# If even this didn't work - we give up - there is an error
				{
				ACTINIC::ReportError($sError, ACTINIC::GetPath());
				}
	
			$ACTINIC::AssertIsActive = $::TRUE;				# Cheat here to make sure that PrintPage doesn't call XML parser
			ACTINIC::PrintPage($sHTML, undef, $::TRUE);	# Print warning page and exit
			exit;

#			return $sFullTag;
			}
		}
	return "";
	}

############################################################
#
#  PriceTagHandler - price tag callback
#
#  Arguments : $sTag - tag name
#              $sInsideText - reference to text between start and end,
#              $ParameterHash - hash of parameters,
#              $sId - current tag prefix,
#              $sFullTag - full text of current tag;
#  Returns   : nothing but replaces $sInsideText by a table of prices
#
#  Ryszard Zybert  Dec  7 21:06:24 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
#
############################################################

sub PriceTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
		
	if( $sTag !~ /^\// )																						# If not End tag
		{
		if( $Self->{CurrentSectionBlob} )																# If section blob set
			{
			my @Response;
			#
			# We need the tax information to calculate the prices
			#
			if(!$ActinicOrder::bTaxDataParsed)
				{
				#
				# read the tax blob
				#
				@Response = ACTINIC::ReadTaxSetupFile(ACTINIC::GetPath());
				if ($Response[0] != $::SUCCESS)
					{
					return (@Response);
					}
				ActinicOrder::ParseAdvancedTax();
				}
			$Self->Parse($$sInsideText);

			@Response = ACTINIC::GetProduct($ParameterHash->{PROD_REF}, $Self->{CurrentSectionBlob},
														  ACTINIC::GetPath());							# get this product object
			my ($Status, $Message, $pProduct) = @Response;
			if ($Status != $::SUCCESS) 
				{ 
				return "";
				}													# If any problem, forget it

			if (defined $$pProduct{PRICES})
				{
				#
				# Need to know if this product has any variants
				#
				my ($VariantList, $sLine);
				if( $pProduct->{COMPONENTS} )
					{
					($VariantList, $sLine) = ACTINIC::GetVariantList($ParameterHash->{PROD_REF});
					}
				#
				# Need to work out which prices to show
				#
				my ($bShowRetailPrices, $bShowCustomerPrices, $nAccountSchedule) = ACTINIC::DeterminePricesToShow();
				
				my $sPriceLabelText = $ACTINIC::B2B->GetXML('RETAIL_PRICE_TEXT');
				if($bShowRetailPrices && $bShowCustomerPrices)
					{
					my $sPriceLabel = ACTINIC::GetPhrase(-1, 294, $sPriceLabelText);
					#
					# Show dealer and retail price
					#
					@Response = ActinicOrder::FormatSchedulePrices($pProduct, $ActinicOrder::RETAILID,
						\$VariantList, $sPriceLabel, 1);								# Show the retail price
					$$sInsideText = $Response[2];
					
					$sPriceLabel = ACTINIC::GetPhrase(-1, 293, $sPriceLabelText);
					@Response = ActinicOrder::FormatSchedulePrices($pProduct, 
						$nAccountSchedule, \$VariantList, $sPriceLabel);			# Show the dealer price
					$$sInsideText .= $Response[2];
					}
				elsif($bShowCustomerPrices)
					{
					#
					# Show only dealer price
					#
					if (0 == scalar(@{$pProduct->{'PRICES'}->{$nAccountSchedule}}))
						{
						#
						# The product is unavailable if user's price schedule is not included
						#
						$$sInsideText = ACTINIC::GetPhrase(-1, 351);	# 'This product is currently unavailable'
						}
					else										# user's price schedule included
						{
						@Response = ActinicOrder::FormatSchedulePrices($pProduct, 
							$nAccountSchedule, \$VariantList, $ACTINIC::B2B->GetXML('RETAIL_PRICE_TEXT'));
						$$sInsideText = $Response[2];
						}
					}
				else
					{
					#
					# Show only retail price
					#
					if (0 == scalar(@{$pProduct->{'PRICES'}->{$ActinicOrder::RETAILID}}))
						{
						#
						# The product is unavailable if the retail price is not included
						#
						$$sInsideText = ACTINIC::GetPhrase(-1, 351);	# 'This product is currently unavailable'
						}
					else
						{
						@Response = ActinicOrder::FormatSchedulePrices($pProduct, 
							1, \$VariantList,  $ACTINIC::B2B->GetXML('RETAIL_PRICE_TEXT'));
						$$sInsideText = $Response[2];
						}
					}
				}
			}
		}
	return "";																									# Always remove tag
	}

############################################################
#  sub SectionTagHandler - section tag callback
#
#  Note: <Actinic:SECTION BLOB="blob file name"/> acts globally
#        $Self->{CurrentSectionBlob} is set here and kept
#
#  Arguments : $sTag - tag name
#              $sInsideText - reference to text between start and end,
#              $ParameterHash - hash of parameters,
#              $sId - current tag prefix,
#              $sFullTag - full text of current tag;
#  Returns   : nothing but sets $Self->{CurrentSectionBlob} to value of parameter BLOB
#
#  Ryszard Zybert  Dec  20 21:06:24 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
############################################################

sub SectionTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId) = @_;
	
	if( $sTag !~ /^\// )														# If not End tag
		{
		$Self->{CurrentSectionBlob} = $ParameterHash->{BLOB};
		}
	return "";																	# Always remove tag
	}

############################################################
#  sub MainFrameTagHandler
#  replace SRC parameter in a FRAME tag
#  
#  If MAINFRAME XML variable is defined and SRC=name is found
#  in inside text, name will be replaced by the value of MAINFRAME XML
#  variable.
#  
#    Arguments : $sTag - tag name
#                $sInsideText - reference to text between start and end,
#                $ParameterHash - hash of parameters,
#                $sId - current tag prefix,
#                $sFullTag - full text of current tag;
#    Returns   : nothing but modifies $$sInsideText
#
#  Ryszard Zybert  Jul 18 11:26:08 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
############################################################

sub MainFrameTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId) = @_;
	
	if( $sTag !~ /^\// )														# If not End tag
		{
		my $sXMLTag;
		if( $::g_InputHash{MAINFRAMEURL} )
			{
			$sXMLTag = $::g_InputHash{MAINFRAMEURL};
			}
		else
			{
			$sXMLTag = $ACTINIC::B2B->GetXML("MAINFRAMEURL");
			}
		if( defined($sXMLTag) )												# Replace SRC parameter by tag value
			{
			if( ref($sInsideText) )
				{
				if( $sXMLTag !~ /^((http:)|(\/))/ )
					{
					if( $sXMLTag eq 'frameset.html' )
						{
						$sXMLTag = 'catalogbody.html'
						}
					$sXMLTag = $::g_sAccountScript . '?' . 'PRODUCTPAGE=' . $sXMLTag;
					}
				$$sInsideText =~ s/(\s+SRC\s*=\s*)((\"[^\"]+\")|([^\ \>]+))((\s+)|(\>+))/$1\"$sXMLTag\"$5/is;
				}
			}
		}
	return "";																	# Always remove tag
	}
############################################################
#  FormatPrice - format single price
#  
#   Arguments: 	$Price - price to display
#						sPriceMsg - price prompt
#  					$sTax - tax info
#  					$sIncTax - including tax info
#  					$bTaxExlusiveOnly - display tax exclusive only
#  					$bTaxInclusiveOnly - display tax inclusive only
#   Returns: formatted price info
#
#  Ryszard Zybert  Dec 29 23:07:05 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
############################################################

sub FormatPrice
	{
	my $Self = shift;
	my ($Price,$sPriceMsg,$sTax,$sIncTax,$bTaxExlusiveOnly,$bTaxInclusiveOnly) = @_;
	my ($sPrice,$sEPrice,$fPrice,$sPriceexl,$sPriceincl);

	my $sCurrency  		= $::g_pCatalogBlob->{SCURRENCY};
	my $sEFormat   		= $::g_pSetupBlob->{EURO_FORMAT};
	my $sECurrency 		= $::g_pCatalogBlob->{EUR}->{SCURRENCY};
	my $fEuroConversion	= $::g_pCatalogBlob->{EUR}->{EXCH_RATE};
	my $sPFormat			= '%s%.2f';

	if( $bTaxExlusiveOnly )																				# Exlusive only
		{
		$fPrice = $Price/100.0;																			# Price in real money
		}			
	else																										# Inclusive or both
		{
		$fPrice = (1.0 + $::g_pSetupBlob->{TAX_1_RATE}/10000.0) * $Price/100.0;			# Add tax
		}
	if( !$bTaxInclusiveOnly and !$bTaxExlusiveOnly )											# Display exclusive and inclusive prices
		{
		$sPriceexl  = sprintf($sPFormat,$sCurrency,$Price/100.0);								# Format exlusive price
		$sPriceincl = sprintf($sPFormat,$sCurrency,$fPrice);										# Format inclusive price
		if( $::g_pSetupBlob->{EURO_PRICES} )															# Euro prices if needed
			{
			$sEPrice    = sprintf($sPFormat,$sECurrency,$Price/$fEuroConversion/100.0);	# Calculate Euro exclusive price
			$sPriceexl  = sprintf($sEFormat,$sPriceexl,$sEPrice);									# Format exclusive price string
			$sEPrice    = sprintf($sPFormat,$sECurrency,$fPrice/$fEuroConversion);			# Calculate Euro inclusive price
			$sPriceincl = sprintf($sEFormat,$sPriceincl,$sEPrice);								# Format inclusive price string
			}
		return ACTINIC::GetPhrase(-1,227,$sPriceMsg,$sPriceexl,$sPriceincl,$sIncTax);
		}

	$sPrice = sprintf($sPFormat,$sCurrency,$fPrice);												# Format price
	if( $::g_pSetupBlob->{EURO_PRICES} )																# Add Euro price if needed
		{
		$sEPrice = sprintf($sPFormat,$sECurrency,$fPrice/$fEuroConversion);					# Calculate Euro price
		$sPrice  = sprintf($sEFormat,$sPrice,$sEPrice);												# Format complete price string
		}
	return ACTINIC::GetPhrase(-1,225,$sPriceMsg,$sPrice,$sTax);
	}
############################################################
#  FormatPriceRow - format price row
#  
#   Arguments: 	$Price - price to display
#  					$sIncTax - including tax info
#						$sQlimit - quantity info
#  					$bTaxExlusiveOnly - display tax exclusive only
#  					$bTaxInclusiveOnly - display tax inclusive only
#   Returns: formatted price info
#
#  Ryszard Zybert  Dec 29 23:07:05 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
############################################################

sub FormatPriceRow
	{
	my $Self = shift;
	my ($Price,$sIncTax,$sQlimit,$bTaxExlusiveOnly,$bTaxInclusiveOnly) = @_;
	my ($sPrice,$sEPrice,$fPrice,$sPriceexl,$sPriceincl);

	my $sCurrency  		= $::g_pCatalogBlob->{SCURRENCY};
	my $sEFormat   		= $::g_pSetupBlob->{EURO_FORMAT};
	my $sECurrency 		= $::g_pCatalogBlob->{EUR}->{SCURRENCY};
	my $fEuroConversion	= $::g_pCatalogBlob->{EUR}->{EXCH_RATE};
	my $sPFormat			= '%s%.2f';

	if( $bTaxExlusiveOnly )																					# Exclusive only
		{
		$fPrice = $Price/100.0;																				# Price in real money
		}	
	else																											# Inclusive or both
		{
		$fPrice = (1.0 + $::g_pSetupBlob->{TAX_1_RATE}/10000.0) * $Price/100.0;				# Add tax
		}

	if( !$bTaxInclusiveOnly and !$bTaxExlusiveOnly )												# Display exclusive and inclusive prices
		{
		$sPriceexl  = sprintf($sPFormat,$sCurrency,$Price/100.0);								# Format exclusive
		$sPriceincl = sprintf($sPFormat,$sCurrency,$fPrice);										# Format inclusive
		if( $::g_pSetupBlob->{EURO_PRICES} )															# Add Euro price if needed
			{
			$sEPrice    = sprintf($sPFormat,$sECurrency,$Price/$fEuroConversion/100.0);	# Exclusive
			$sPriceexl  = sprintf($sEFormat,$sPriceexl,$sEPrice);
			$sEPrice    = sprintf($sPFormat,$sECurrency,$fPrice/$fEuroConversion);			# Inclusive
			$sPriceincl = sprintf($sEFormat,$sPriceincl,$sEPrice);
			}
		if( $Self->{Variables}->{FORMAT_PRICE_ROW_BOTH} )											# If there is a variable, use it
			{
			return sprintf($Self->{Variables}->{FORMAT_PRICE_ROW_BOTH},$sPriceexl,$sPriceincl,$sIncTax,$sQlimit);
			}	
		else																										# Otherwise use prompt 228
			{
			return ACTINIC::GetPhrase(-1,228,$sPriceexl,$sPriceincl,$sIncTax,$sQlimit);
			}
		}

	$sPrice = sprintf($sPFormat,$sCurrency,$fPrice);												# Display either exclusive or inclusive
	if( $::g_pSetupBlob->{EURO_PRICES} )																# Add Euro prices if needed
		{
		$sEPrice = sprintf($sPFormat,$sECurrency,$fPrice/$fEuroConversion);
		$sPrice  = sprintf($sEFormat,$sPrice,$sEPrice);
		}
	if( $Self->{Variables}->{FORMAT_PRICE_ROW} )														# If there is a variable, use it
		{
		return sprintf($Self->{Variables}->{FORMAT_PRICE_ROW},$sPrice,$sQlimit);
		}
	else																											# Otherwise use prompt 224
		{
		return ACTINIC::GetPhrase(-1,224,$sPrice,$sQlimit);
		}
	}

1;
